<?php

namespace Tests\Unit;

use App\Enums\DiaSemanaEnum;
use App\Enums\StatusAgendamentoAtendimentoEnum;
use App\Enums\TipoAgendamentoEnum;
use App\Enums\TipoPrestadorServicoEnum;
use App\Exceptions\AgendamentoNaoPodeSerCanceladoException;
use App\Models\AgendamentoAtendimento;
use App\Models\HorarioTrabalho;
use App\Models\MotivoCancelamentoAgendamento;
use App\Models\PrestadorServico;
use App\Models\ServicoAtendimento;
use App\Models\Servidor;
use App\Services\AgendaAtendimentoJuridicoService;
use App\Services\AgendaAtendimentoService;
use App\Services\CancelaAgendamentoAtendimentoService;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase as TestCase;

class CancelaAgendamentoAtendimentoServiceTest extends TestCase
{
    use RefreshDatabase;

    public function test_cancela()
    {
        $pessoa = factory(Servidor::class)->create();
        $motivo = factory(MotivoCancelamentoAgendamento::class)->create();

        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::segunda()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');
        $serviceAgendamento = new AgendaAtendimentoJuridicoService($inicio, $fim, $pessoa, $servico);
        $agendamento = $serviceAgendamento->agenda();
        $this->assertInstanceOf(AgendamentoAtendimento::class, $agendamento);
        $this->assertTrue($agendamento->status->isAberto());

        $service = new CancelaAgendamentoAtendimentoService($agendamento, $motivo, 'Teste');
        $agendamento = $service->cancela();
        $this->assertInstanceOf(AgendamentoAtendimento::class, $agendamento);
        $this->assertNotNull($agendamento->motivo_cancelamento_id);
        $this->assertEquals('Teste', $agendamento->observacao_cancelamento);
        $this->assertTrue($agendamento->status->isDesmarcou());
    }

    public function test_cancela_agendamento_desmarcou()
    {
        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::segunda()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');
        $serviceAgendamento = new AgendaAtendimentoJuridicoService($inicio, $fim, $pessoa, $servico);
        $agendamento = $serviceAgendamento->agenda();
        $this->assertInstanceOf(AgendamentoAtendimento::class, $agendamento);

        $agendamento->status = StatusAgendamentoAtendimentoEnum::desmarcou();
        $agendamento->save();

        $motivo = factory(MotivoCancelamentoAgendamento::class)->create();
        $service = new CancelaAgendamentoAtendimentoService($agendamento, $motivo, 'Teste');
        $this->expectException(AgendamentoNaoPodeSerCanceladoException::class);
        $service->cancela();
    }
}
