<?php

namespace App\Models;

use App\Traits\Contatavel;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class UnidadeOrganizacional extends Model
{
    use Contatavel;

    protected $table = 'unidades_organizacionais';
    protected $guarded = ['id', 'deleted_at'];
    protected $appends = ['sigla_nome'];

    public function superior()
    {
        return $this->belongsTo(self::class, 'superior_id', 'id');
    }

    public function subordinada()
    {
        return $this->hasMany(self::class, 'superior_id');
    }

    public function registrosServidores()
    {
        return $this->hasMany(RegistroServidor::class, 'unidadeorganizacional_id');
    }

    public function scopeSuperiores($query)
    {
        return $query->where('superior_id', null);
    }

    public function hierarquiaSuperior()
    {
        $superiores = $this->where('id', '=', $this->superior_id)->get();

        while ($superiores->last() && $superiores->last()->superior_id !== null) {
            $superior = $this->where('id', '=', $superiores->last()->superior_id)->get();
            $superiores = $superiores->merge($superior);
        }

        return $superiores;
    }

    public function hierarquiaInferior()
    {
        $unidades = collect();

        foreach ($this->subordinada as $subordinada) {
            $unidades->push($subordinada);
            $unidades = $unidades->merge($subordinada->hierarquiaInferior());
        }

        return $unidades;
    }

    public function enderecos()
    {
        return $this->morphMany(Endereco::class, 'enderecavel');
    }

    public function secretaria()
    {
        $hierarquiaSuperior = $this->hierarquiaSuperior();

        // Prefeitura na maioria dos casos, mas vamos generalizar
        // Remove da coleção
        $hierarquiaSuperior->pop();

        return $hierarquiaSuperior->last();
    }

    public function empresa()
    {
        $hierarquiaSuperior = $this->hierarquiaSuperior();

        return $hierarquiaSuperior->last();
    }

    public function isSuperior()
    {
        return !$this->superior_id;
    }

    public function temSubordinada()
    {
        return (bool) $this->subordinada()->count();
    }

    public function temServidores()
    {
        return (bool) $this->registrosServidores()->count();
    }

    public function scopeSearch($query, $busca = '')
    {
        if (!$busca) {
            $query->superiores();
        } else {
            $query->where('nome', 'ilike', "%{$busca}%");
            $query->orWhere('sigla', 'ilike', "%{$busca}%");
        }

        return $query->orderBy('nome', 'asc');
    }

    public function scopeDeNivel($query, int $nivel)
    {
        return $query->where('nivel', $nivel);
    }

    public function scopeSecretarias($query)
    {
        return $query->deNivel(config('app.nivel-secretaria'));
    }

    public function hierarquiaExtenso()
    {
        if (!$this->hierarquia) {
            return '';
        }

        $ids = explode('/', $this->hierarquia);
        $unidades = UnidadeOrganizacional::whereIn('id', $ids)->get();

        return $unidades->pluck('sigla')->implode(' / ');
    }

    public function transferePara(UnidadeOrganizacional $unidadeSuperior = null)
    {
        if (!$unidadeSuperior) {
            return $this->transfereParaRaiz();
        }

        DB::beginTransaction();

        $this->nivel = $unidadeSuperior->nivel + 1;
        $this->hierarquia = "{$unidadeSuperior->hierarquia}/{$this->id}";
        $this->superior()->associate($unidadeSuperior);
        $this->save();

        $hierarquiaInferior = $this->hierarquiaInferior();

        foreach ($hierarquiaInferior as $unidade) {
            $unidade->hierarquia = "{$unidade->superior->hierarquia}/{$unidade->id}";
            $unidade->save();
        }

        DB::commit();
    }

    private function transfereParaRaiz()
    {
        DB::beginTransaction();

        $this->nivel = 1;
        $this->hierarquia = $this->id;
        $this->superior_id = null;
        $this->save();

        $hierarquiaInferior = $this->hierarquiaInferior();

        foreach ($hierarquiaInferior as $unidade) {
            $unidade->hierarquia = "{$unidade->superior->hierarquia}/{$unidade->id}";
            $unidade->save();
        }

        DB::commit();
    }

    public function getSiglaNomeAttribute()
    {
        return $this->attributes['sigla'] . ' - ' . $this->attributes['nome'];
    }
}
