<?php

namespace App\Models;

use App\Enums\TipoBeneficiarioConvenioEnum;
use App\Enums\TipoConvenioEnum;
use App\Traits\Contatavel;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;

class Convenio extends Model
{
    use Contatavel;

    protected $casts = [
        'desconto' => 'boolean',
    ];
    protected $fillable = ['nome', 'cpf', 'cnpj', 'validade', 'desconto', 'especialidade_id', 'observacoes'];

    public function endereco()
    {
        return $this->morphOne(Endereco::class, 'enderecavel')->withDefault();
    }

    public function especialidade()
    {
        return $this->belongsTo(EspecialidadeConvenio::class);
    }

    public function dependentes()
    {
        return $this->hasMany(Dependente::class);
    }

    public function setTipoAttribute(TipoConvenioEnum $tipo)
    {
        $this->attributes['tipo'] = $tipo->getIndex() ?? null;
    }

    public function getTipoAttribute()
    {
        if (! $this->attributes['tipo']) {
            return null;
        }
        
        return TipoConvenioEnum::make($this->attributes['tipo']);
    }

    public function setTipoBeneficiarioAttribute(TipoBeneficiarioConvenioEnum $tipo)
    {
        $this->attributes['tipo_beneficiario'] = $tipo->getIndex() ?? null;
    }

    public function getTipoBeneficiarioAttribute()
    {
        if (! $this->attributes['tipo_beneficiario']) {
            return null;
        }
        
        return TipoBeneficiarioConvenioEnum::make($this->attributes['tipo_beneficiario']);
    }

    public function setValidadeAttribute($data)
    {
        $this->attributes['validade'] = $data ? Carbon::createFromFormat('d/m/Y', $data) : null;
    }

    public function getValidadeAttribute()
    {
        return Carbon::create($this->attributes['validade']);
    }

    public function setCpfAttribute($cpf)
    {
        $this->attributes['cpf'] = str_replace(['.', '-'], '', $cpf);
    }

    public function setCnpjAttribute($cnpj)
    {
        $this->attributes['cnpj'] = str_replace(['.', '-', '/'], '', $cnpj);
    }

    public function scopeBuscaPorTermo($query, $termo = '')
    {
        if ($termo) {
            $query->where('nome', 'ilike', "%{$termo}%");
        }

        return $query->orderBy('nome');
    }

    public function eOsan()
    {
        return $this->attributes['id'] == 1;
    }

    public function eSeguro()
    {
        return $this->attributes['id'] == 2;
    }

    public function scopeOsan($query)
    {
        return $query->where('id', 1);
    }

    public function scopeSeguro($query)
    {
        return $query->where('id', 2);
    }

    public function scopeParaAssociadosEDependentes($query)
    {
        return $query->where('tipo_beneficiario', TipoBeneficiarioConvenioEnum::associados_dependentes()->getIndex());
    }

    public function scopeParaAssociados($query)
    {
        return $query->where('tipo_beneficiario', TipoBeneficiarioConvenioEnum::associados()->getIndex());
    }
}
