<?php

namespace App\Http\Requests;

use App\Enums\RegimeServidorEnum;
use App\Models\RegistroServidor;
use Illuminate\Foundation\Http\FormRequest;

class RegistroServidorUpdateRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    protected function prepareForValidation(): void
    {
        $this->merge([
            'cep' => str_replace(['.', '-'], '', $this->cep)
        ]);
    }


    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'numero_registro' => ['required', 'numeric'],
            'regime' => 'required|numeric',
            'data_admissao' => 'required|date_format:d/m/Y',
            'data_desligamento' => 'nullable|date_format:d/m/Y|after_or_equal:data_admissao',
            'cargo_id' => 'required|numeric',
            'readaptado' => 'required|boolean',
            'cargoreadaptado_id' => 'required_if:readaptado,1',
            'unidadeorganizacional_id' => 'required|numeric',

            'paridade' => "nullable|boolean",
            'proventos' => "nullable|numeric",
            'tipo_aposentadoria' => "nullable|numeric",
            'data_aposentadoria' => "nullable|date|after:data_admissao",
            'portaria_aposentadoria' => "nullable|string|max:20",

            'usar_endereco_unidade' => 'required|boolean',
            'cep' => 'required_if:usar_endereco_unidade,0|numeric',
            'logradouro' => 'required_with:cep|max:150',
            'bairro' => 'required_with:cep|max:50',
            'cidade' => 'required_with:cep|max:100',
            'numero' => 'required_with:cep|nullable|numeric',
            'estado' => 'required_with:cep|max:2',
            'observacoes_endereco' => 'nullable|max:100',
        ];
    }

    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            if ($this->registroJaExiste()) {
                $validator->errors()->add('registro', 'Já existe um registro com o número e regime informados!');
            }
        });
    }

    private function registroJaExiste()
    {
        return RegistroServidor::where('numero', $this->numero_registro)
                                    ->where('regime', $this->regime)
                                    ->where('id', '!=', $this->registro->id)->count() > 0;
    }

    public function attributes()
    {
        return [
            'numero' => 'número',
            'numero_registro' => 'número',
            'data_admissao' => 'data de admissão',
            'unidadeorganizacional_id' => 'unidade organizacional',
            'cep' => 'CEP',
            'observacoes_endereco' => 'observações do endereço',
            'data_aposentadoria' => 'data de aposentadoria',
            'portaria_aposentadoria' => 'portaria de aposentadoria',
        ];
    }

    public function messages()
    {
        return [
            'data_nascimento.before' => 'O campo data de nascimento deve ser uma data anterior à data atual.',
            'data_admissao.after' => 'O campo data de admissão deve ser uma data posterior à data de nascimento.',
            'data_desligamento.after_or_equal' => 'O campo data de desligamento deve ser uma data posterior ou igual à data de admissão.',
            'cargoreadaptado_id.required_if' => 'O campo cargo readaptado é obrigatório quando o servidor é readaptado.',
            'cep.required_if' => 'O endereço é obrigatório quando não se utiliza o endereço da unidade.',
            'paridade.required_if' => 'O campo Paridade é obrigatório quando o servidor for aposentado',
            'proventos.required_if' => "O campo Proventos é obrigatório quando o servidor for aposentado",
            'tipo_aposentadoria.required_if' => "O campo tipo de aposentadoria é obrigatório quando o servidor for aposentado",
            'data_aposentadoria.required_if' => "O campo data de aposentadoria é obrigatório quando o servidor for aposentado",
            'portaria_aposentadoria.required_if' => "O campo portaria é obrigatório quando o servidor for aposentado",
        ];
    }
}
