<?php

namespace App\Http\Controllers\Admin\Servidor;

use App\Enums\MotivoExclusaoConvenioEnum;
use App\Http\Controllers\Controller;
use App\Models\Convenio;
use App\Models\ConvenioPessoa;
use App\Models\Dependente;
use App\Models\Servidor;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ConvenioDependenteController extends Controller
{
    public function index(Servidor $servidor, Dependente $dependente)
    {
        return view('admin.servidores.convenios.dependentes.index', [
            'servidor' => $servidor,
            'dependente' => $dependente,
            'convenios' => $dependente->convenios,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Servidor $servidor, Dependente $dependente)
    {
        $conveniosDependente = $dependente->convenios->pluck('convenio_id');
        $convenios = Convenio::paraAssociadosEDependentes()->whereNotIn('id', $conveniosDependente)->orderBy('nome')->get();

        return view('admin.servidores.convenios.dependentes.create', [
            'servidor' => $servidor,
            'dependente' => $dependente,
            'convenios' => $convenios
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, Servidor $servidor, Dependente $dependente)
    {
        $convenio = new ConvenioPessoa($request->all());
        $dependente->convenios()->save($convenio);

        return redirect()
            ->route('admin.servidores.dependentes.convenios.index', ['servidor' => $servidor, 'dependente' => $dependente])
            ->with('status', 'Convênio incluído com sucesso!');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Convenio  $convenio
     * @return \Illuminate\Http\Response
     */
    public function edit(Servidor $servidor, Dependente $dependente, ConvenioPessoa $convenio)
    {
        $conveniosDependente = $dependente->convenios->where('convenio_id', '!=', $convenio->convenio->id)->pluck('convenio_id');
        $convenios = Convenio::paraAssociadosEDependentes()->whereNotIn('id', $conveniosDependente)->orderBy('nome')->get();

        return view('admin.servidores.convenios.dependentes.edit', [
            'servidor' => $servidor,
            'dependente' => $dependente,
            'convenios' => $convenios,
            'convenio' => $convenio,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Convenio  $convenio
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Servidor $servidor, Dependente $dependente, ConvenioPessoa $convenio)
    {
        $convenio->fill($request->all());
        $convenio->save();

        return redirect()
            ->route('admin.servidores.dependentes.convenios.index', ['servidor' => $servidor, 'dependente' => $dependente])
            ->with('status', 'Convênio editado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Convenio  $convenio
     * @return \Illuminate\Http\Response
     */
    public function destroy(Servidor $servidor, Dependente $dependente, ConvenioPessoa $convenio)
    {
        $possuiTermos = $dependente->convenios()->temOsanOuSeguro()->first();

        DB::beginTransaction();
        $convenio->motivo_exclusao = MotivoExclusaoConvenioEnum::outro();
        $convenio->save();
        $convenio->delete();
        DB::commit();

        $redirect = redirect()
            ->route('admin.servidores.dependentes.convenios.index', ['servidor' => $servidor, 'dependente' => $dependente])
            ->with('status', 'Convênio excluído com sucesso!');

        if ($convenio->convenio->eOsan()) {
            return $redirect->with('cancelou-osan', true);
        }

        if ($possuiTermos) {
            $redirect->with('termos-assinar', true);
        }

        return $redirect;
    }
}
