<?php

namespace App\Http\Controllers\Admin;

use App\Enums\EspecialidadeOdontologiaEnum;
use App\Enums\SexoEnum;
use App\Enums\TipoPrestadorServicoEnum;
use App\Http\Controllers\Controller;
use App\Http\Requests\PrestadorServicoStoreRequest;
use App\Http\Requests\PrestadorServicoUpdateRequest;
use App\Models\Endereco;
use App\Models\Logradouro\Estado;
use App\Models\PrestadorServico;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PrestadorServicoController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $termo = $request->busca;
        $prestadores = PrestadorServico::with('contatos')
            ->buscaPorTermo($termo)
            ->paginate();
        

        return view('admin.prestadores-servicos.index', [
            'prestadores' => $prestadores,
            'termo' => $termo,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $sexos = SexoEnum::getAll();
        $tipos = TipoPrestadorServicoEnum::getAll();
        $especialidades = EspecialidadeOdontologiaEnum::getAll();
        $estados = Estado::orderBy('ufe_sg')->select('ufe_sg')->get();

        return view('admin.prestadores-servicos.create', [
            'sexos' => $sexos,
            'tipos' => $tipos,
            'especialidades' => $especialidades,
            'estados' => $estados
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(PrestadorServicoStoreRequest $request)
    {
        $prestador = new PrestadorServico($request->only('nome', 'funcao', 'rg', 'cpf', 'cnpj', 'observacoes'));
        $prestador->sexo = $request->sexo ? SexoEnum::make((int) $request->sexo) : null;
        $prestador->tipo = $request->tipo ? TipoPrestadorServicoEnum::make((int) $request->tipo) : null;
        $prestador->especialidade = $request->especialidade ? EspecialidadeOdontologiaEnum::make((int) $request->especialidade) : null;
        
        DB::beginTransaction();

        $prestador->save();
        $prestador->saveTelefones($request->telefone);

        if ($request->cep) {
            $prestador->endereco->fill($request->only('cep', 'logradouro', 'numero', 'cidade', 'bairro', 'estado'));
            $prestador->endereco->observacoes = $request->observacoes_endereco;
            $prestador->endereco->save();
        }

        DB::commit();

        return redirect()
            ->route('admin.prestadores-servicos.edit', ['prestador' => $prestador])
            ->with('sucesso', true)
            ->with('status', 'Prestador de serviço incluído com sucesso!');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\PrestadorServico  $prestador
     * @return \Illuminate\Http\Response
     */
    public function edit(PrestadorServico $prestador)
    {
        $sexos = SexoEnum::getAll();
        $tipos = TipoPrestadorServicoEnum::getAll();
        $especialidades = EspecialidadeOdontologiaEnum::getAll();
        $estados = Estado::orderBy('ufe_sg')->select('ufe_sg')->get();
        
        return view('admin.prestadores-servicos.edit', [
            'prestador' => $prestador,
            'sexos' => $sexos,
            'tipos' => $tipos,
            'especialidades' => $especialidades,
            'estados' => $estados
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\PrestadorServico  $prestador
     * @return \Illuminate\Http\Response
     */
    public function update(PrestadorServicoUpdateRequest $request, PrestadorServico $prestador)
    {
        $prestador->fill($request->only('nome', 'funcao', 'rg', 'cpf', 'cnpj', 'observacoes'));
        $prestador->sexo = $request->sexo ? SexoEnum::make((int) $request->sexo) : null;
        $prestador->tipo = $request->tipo ? TipoPrestadorServicoEnum::make((int) $request->tipo) : null;
        $prestador->especialidade = $request->especialidade ? EspecialidadeOdontologiaEnum::make((int) $request->especialidade) : null;

        $prestador->save();
        $prestador->saveTelefones($request->telefone ?? []);
        
        if ($request->cep) {
            $prestador->endereco->fill($request->only('cep', 'logradouro', 'numero', 'cidade', 'bairro', 'estado'));
            $prestador->endereco->observacoes = $request->observacoes_endereco;
            $prestador->endereco->save();
        } else {
            $prestador->endereco->delete();
        }

        DB::commit();

        return redirect()
            ->back()
            ->with('sucesso', true)
            ->with('status', 'Prestador de serviço editado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\PrestadorServico  $prestador
     * @return \Illuminate\Http\Response
     */
    public function destroy(PrestadorServico $prestador)
    {
        $prestador->delete();

        return redirect()->back()
                ->with('status', 'Prestador de serviço excluído com sucesso!');
    }
}
