<?php

namespace App\Http\Controllers\Admin;

use App\Enums\FiltroBuscaServidorEnum;
use App\Enums\SituacaoServidorBuscaEnum;
use App\Http\Controllers\Controller;
use App\Models\RegistroAtendimento;
use App\Models\Servidor;
use App\Models\MotivoRegistroAtendimento;
use Illuminate\Http\Request;

class GuiaAtendimentoController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth']);
    }

    public function buscaServidor(Request $request)
    {
        $servidores = Servidor::buscaPorTermo($request->busca)->get();

        return $servidores->map(function ($servidor) {
            return [
                'text' => $servidor->nome,
                'id' => $servidor->id,
            ];
        });
    }

    public function index(Request $request)
    {
        if ($request->servidor) {
            $servidor = Servidor::find($request->servidor);
            $atendimentos = $servidor->registrosAtendimentos();
        } else {
            $atendimentos = RegistroAtendimento::query();
        }

        $atendimentos->with('motivo');

        if ($request->input('resolvido', '') != '') {
            $atendimentos->resolvidos($request->resolvido);
        }
        
        return view('admin.guia-atendimento.index', [
            'busca' => $request->busca ?? null,
            'atendimentos' => $atendimentos->orderBy('id', 'desc')->paginate()
        ]);
    }

    public function buscar(Request $request)
    {
        $servidores = collect();

        if ($request->has('filtro')) {
            $filtro = FiltroBuscaServidorEnum::make((int) $request->filtro);
        }
        
        if ($request->busca) {
            $servidores = Servidor::busca($filtro, $request->busca, SituacaoServidorBuscaEnum::todos(), true)->paginate();
        }

        return view('admin.guia-atendimento.busca', [
            'servidores' => $servidores,
            'filtros' => FiltroBuscaServidorEnum::getAll(),
            'filtroSelecionado' => $filtro ?? FiltroBuscaServidorEnum::matricula(),
            'busca' => $request->busca ?? null
        ]);
    }

    public function create(Servidor $servidor)
    {
        $motivos = MotivoRegistroAtendimento::all();
        return view('admin.guia-atendimento.create', ['servidor' => $servidor, 'motivos' => $motivos]);
    }

    public function store(Servidor $servidor, Request $request)
    {
        $this->validate($request, [
            'atendimento' => 'required|string',
            'resolvido' => 'required|boolean',
        ]);

        $atendimento = new RegistroAtendimento($request->all());
        $atendimento->usuario()->associate($request->user());
        $atendimento->motivo()->associate(MotivoRegistroAtendimento::find($request->motivo_id));
        $servidor->registrosAtendimentos()->save($atendimento);

        return redirect()
                    ->route('admin.guia-atendimento.index')
                    ->with('status', 'Registro de atendimento incluído com sucesso!');
    }

    public function edit(RegistroAtendimento $atendimento)
    {
        $motivos = MotivoRegistroAtendimento::all();
        return view('admin.guia-atendimento.edit', [
            'atendimento' => $atendimento,
            'motivos' => $motivos,
        ]);
    }

    public function update(RegistroAtendimento $atendimento, Request $request)
    {
        $this->validate($request, [
            'atendimento' => 'required|string',
            'resolvido' => 'required|boolean',
        ]);

        $atendimento->fill($request->all());
        $atendimento->motivo()->associate(MotivoRegistroAtendimento::find($request->motivo_id));
        $atendimento->save();

        return redirect()
            ->route('admin.guia-atendimento.index')
            ->with('status', 'Registro de atendimento editado com sucesso!');
    }
}
