<?php

namespace App\Http\Controllers\Admin\Funcionario;

use App\Enums\MotivoExclusaoConvenioEnum;
use App\Http\Controllers\Controller;
use App\Models\Convenio;
use App\Models\ConvenioPessoa;
use App\Models\Dependente;
use App\Models\Funcionario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ConvenioDependenteController extends Controller
{
    public function index(Funcionario $funcionario, Dependente $dependente)
    {
        return view('admin.funcionarios.convenios.dependentes.index', [
            'funcionario' => $funcionario,
            'dependente' => $dependente,
            'convenios' => $dependente->convenios
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Funcionario $funcionario, Dependente $dependente)
    {
        $conveniosDependente = $dependente->convenios->pluck('convenio_id');
        $convenios = Convenio::paraAssociadosEDependentes()->whereNotIn('id', $conveniosDependente)->orderBy('nome')->get();

        return view('admin.funcionarios.convenios.dependentes.create', [
            'funcionario' => $funcionario,
            'dependente' => $dependente,
            'convenios' => $convenios
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, Funcionario $funcionario, Dependente $dependente)
    {
        $convenio = new ConvenioPessoa($request->all());
        $dependente->convenios()->save($convenio);

        return redirect()
            ->route('admin.funcionarios.dependentes.convenios.index', ['funcionario' => $funcionario, 'dependente' => $dependente])
            ->with('status', 'Convênio incluído com sucesso!');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Convenio  $convenio
     * @return \Illuminate\Http\Response
     */
    public function edit(Funcionario $funcionario, Dependente $dependente, ConvenioPessoa $convenio)
    {
        $conveniosDependente = $dependente->convenios()
                                          ->where('convenio_id', '!=', $convenio->convenio->id)
                                          ->get()
                                          ->pluck('convenio_id');
        $convenios = Convenio::paraAssociadosEDependentes()->whereNotIn('id', $conveniosDependente)->orderBy('nome')->get();

        return view('admin.funcionarios.convenios.dependentes.edit', [
            'funcionario' => $funcionario,
            'dependente' => $dependente,
            'convenios' => $convenios,
            'convenio' => $convenio,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Convenio  $convenio
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Funcionario $funcionario, Dependente $dependente, ConvenioPessoa $convenio)
    {
        $convenio->fill($request->all());
        $convenio->save();

        return redirect()
            ->route('admin.funcionarios.dependentes.convenios.index', ['funcionario' => $funcionario, 'dependente' => $dependente])
            ->with('status', 'Convênio editado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Convenio  $convenio
     * @return \Illuminate\Http\Response
     */
    public function destroy(Funcionario $funcionario, Dependente $dependente, ConvenioPessoa $convenio)
    {
        $possuiTermos = $dependente->convenios()->temOsanOuSeguro()->first();

        DB::beginTransaction();
        $convenio->motivo_exclusao = MotivoExclusaoConvenioEnum::outro();
        $convenio->save();
        $convenio->delete();
        DB::commit();
        
        $redirect = redirect()
            ->route('admin.funcionarios.dependentes.convenios.index', ['funcionario' => $funcionario, 'dependente' => $dependente])
            ->with('status', 'Convênio excluído com sucesso!');

        if ($possuiTermos) {
            $redirect->with('termos-assinar', true);
        }

        return $redirect;
    }
}
