<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Dependente;
use App\Models\MensalidadeNivel;
use App\Models\Servidor;

class FormularioController extends Controller
{
    public function __construct()
    {
        $this->middleware(['role:admin']);
    }

    public function desfiliacao(Servidor $servidor)
    {
        $registro = $servidor->registros()->first();
        $unidadeOrganizacional = $registro->unidadeOrganizacional;
        $empresa = optional(optional($unidadeOrganizacional)->empresa())->nome;

        $secretaria = optional(optional($unidadeOrganizacional)->secretaria())->nome;
        $codigoMensalidade = $registro->regime->isEstatutario() ? '348' : '355';

        return view('admin.impressos.formulario-desligamento-sindicato', [
            'servidor' => $servidor,
            'registro' => $registro,
            'secretaria' => $secretaria,
            'empresa' => $empresa,
            'codigoMensalidade' => $codigoMensalidade,
        ]);
    }

    /**
     * Impresso no desligamento do servidor.
     */
    public function desfiliacaoRenunciaPoderesAdvogados(Servidor $servidor)
    {
        return view('admin.impressos.formulario-desligamento-sindicato-renuncia-poderes-advogados', [
            'servidor' => $servidor,
        ]);
    }

    /**
     * Impresso no desligamento do servidor.
     */
    public function cancelamentoSeguro(Servidor $servidor)
    {
        $registro = $servidor->registros()->first();
        $unidadeOrganizacional = $registro->unidadeOrganizacional;
        $secretaria = optional(optional($unidadeOrganizacional)->secretaria())->nome;
        $empresa = optional(optional($unidadeOrganizacional)->empresa())->nome;

        $codigoMensalidade = $registro->regime->isEstatutario() ? '348' : '355';

        return view('admin.impressos.formulario-cancelamento-seguro', [
            'servidor' => $servidor,
            'registro' => $registro,
            'secretaria' => $secretaria,
            'empresa' => $empresa,
            'codigoMensalidade' => $codigoMensalidade,
        ]);
    }

    /**
     * Impresso no desligamento do servidor.
     */
    public function cancelamentoOsan(Servidor $servidor)
    {
        $registro = $servidor->registros()->first();
        $unidadeOrganizacional = $registro->unidadeOrganizacional;
        $secretaria = optional(optional($unidadeOrganizacional)->secretaria())->nome;
        $empresa = optional(optional($unidadeOrganizacional)->empresa())->nome;

        $codigoMensalidade = 506;
        if ($registro->regime->isCeletista()) {
            $codigoMensalidade = 423;
        } else if ($registro->regime->isCamara()) {
            $codigoMensalidade = 346;
        }

        return view('admin.impressos.formulario-cancelamento-osan', [
            'servidor' => $servidor,
            'registro' => $registro,
            'secretaria' => $secretaria,
            'empresa' => $empresa,
            'codigoMensalidade' => $codigoMensalidade,
        ]);
    }


    /**
     * Impresso ao cancelar OSAN do dependente do servidor.
     */
    public function cancelamentoOsanDependente(Servidor $servidor, Dependente $dependente)
    {
        $registro = $servidor->registros()->first();
        $unidadeOrganizacional = $registro->unidadeOrganizacional;
        $secretaria = optional(optional($unidadeOrganizacional)->secretaria())->nome;
        $empresa = optional(optional($unidadeOrganizacional)->empresa())->nome;
        if ($registro->regime->isAposentado()) {
            $empresa = 'IPREV';
        }

        return view('admin.impressos.formulario-cancelamento-osan-dependente', [
            'servidor' => $servidor,
            'dependente' => $dependente,
            'registro' => $registro,
            'secretaria' => $secretaria,
            'empresa' => $empresa,
        ]);
    }

    public function fichaInscricao(Servidor $servidor)
    {
        $registro = $servidor->registros()->first();
        $cargo = $registro->cargo;
        $setor = $registro->unidadeOrganizacional;
        $telefonesSetor = optional(optional(optional($setor)->contatos())->telefonesFixos());
        $telefoneSetor = optional($telefonesSetor->first())->valor;

        if (optional($setor)->nivel == config('app.nivel-secretaria')) {
            $secretaria = optional($setor)->nome;
        } else {
            $secretaria = optional(optional($setor)->secretaria())->nome;
        }

        $matricula = $servidor->matriculas()->latest()->first();
        $numeroMatricula = optional($matricula)->id;
        $dataAdmissaoSindical = optional($matricula)->data_filiacao;
        $empresa = optional(optional($setor)->empresa())->nome;
        $endereco = $servidor->enderecos()->first();
        $enderecoRegistro = $registro->enderecos()->first();
        $enderecoUnidadeOrganizacional = optional(optional($registro->unidadeOrganizacional)->enderecos())->first();
        $enderecoLotacao = $enderecoRegistro ?? $enderecoUnidadeOrganizacional;
        $telefoneFixo = $servidor->contatos()->telefonesFixos()->first();
        $telefoneCelular = $servidor->contatos()->telefonesCelulares()->first();
        $email = optional($servidor->contatos()->emails()->first())->valor;
        $dependentes = $servidor->dependentes;

        if ($registro->nivelSalarial) {
            $nivelSalarial = $registro->nivelSalarial->nome;
            $mensalidade = MensalidadeNivel::vigente($registro->nivelSalarial);
        } else {
            $mensalidade = $cargo ? MensalidadeNivel::vigente($cargo->nivelSalarial) : null;
            $nivelSalarial = optional(optional($cargo)->nivelSalarial)->nome;
        }

        return view('admin.impressos.ficha-inscricao', [
            'servidor' => $servidor,
            'numeroMatricula' => $numeroMatricula,
            'dataAdmissaoSindical' => $dataAdmissaoSindical,
            'enderecoServidor' => $endereco,
            'registro' => $registro,
            'enderecoLotacao' => $enderecoLotacao,
            'cargo' => $cargo,
            'nivelSalarial' => $nivelSalarial,
            'mensalidade' => $mensalidade,
            'mesDesconto' => $this->mesDesconto(),
            'secretaria' => $secretaria,
            'nomeSetor' => optional($setor)->nome,
            'dependentes' => $dependentes,
            'telefoneSetor' => $telefoneSetor,
            'empresa' => $empresa,
            'telefoneFixo' => $telefoneFixo,
            'telefoneCelular' => $telefoneCelular,
            'email' => $email,
        ]);
    }

    public function declaracaoAcimaIdadeLimite(Servidor $servidor)
    {
        $registro = $servidor->registros()->first();

        return view('admin.impressos.declaracao-acima-idade-limite', [
            'servidor' => $servidor,
            'registro' => $registro,
        ]);
    }

    private function mesDesconto()
    {
        $dia = today()->day;

        if ($dia < 5) {
            $primeiroDia = today()->setDay(1);
            $segundoDia = today()->setDay(2);

            if ($dia <= 2 && today()->isWeekday()) {
                return today()->monthName;
            } elseif ($dia <= 3 && $primeiroDia->isSunday()) {
                // Dia 1 domingo
                return today()->monthName;
            } elseif ($dia <= 4 && $segundoDia->isSaturday()) {
                // Dia 2 sábado
                return today()->monthName;
            } elseif ($dia <= 4 && $primeiroDia->isSaturday() && $segundoDia->isSunday()) {
                // Dia 1 sábado, dia 2 domingo
                return today()->monthName;
            } else {
                return today()->addMonth()->monthName;
            }
        } else {
            return today()->addMonth()->monthName;
        }
    }
}
