<?php

namespace App\Http\Controllers\Admin;

use App\Enums\FiltroEnvioMensagemSMSEnum;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Cargo;
use App\Models\MalaDireta;
use App\Models\Servidor;
use App\Models\UnidadeOrganizacional;
use App\Notifications\MalaDiretaNotification;
use Illuminate\Support\Facades\Notification;

class EnvioSMSController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }
    
    public function mensagensEnviadas(Request $request)
    {
        $q = $request->q;
        $mensagens = MalaDireta::with('audits')
                    ->orderBy('created_at', 'desc')
                    ->paginate();

        return view('admin.envio-sms.mensagens-enviadas', compact('mensagens', 'q'));
    }

    public function selecaoContatosForm(Request $request)
    {
        $filtroServidor = $request->filtro_servidor ?? 'todos';
        $filtro = FiltroEnvioMensagemSMSEnum::make($filtroServidor);
        $request->session()->put('mensagens.filtroServidor', $filtro);

        $contatos = Servidor::with('registros.unidadeOrganizacional', 'registros.cargo')
            ->associado()
            ->naoPerturbe(false)
            ->comCelularMalaDireta();

        if ($request->cargo) {
            $cargo = Cargo::find($request->cargo);
            $contatos->comCargo($cargo);
        }

        if ($request->secretaria) {
            $secretaria = UnidadeOrganizacional::find($request->secretaria);
            $contatos->alocadosNaSecretaria($secretaria);
        }

        return view('admin.envio-sms.selecao-contatos', [
            'contatos' => $contatos->orderBy('nome')->get(),
            'filtroServidor' => $filtroServidor,
            'cargos' => Cargo::orderBy('nome')->get(),
            'filtros' => FiltroEnvioMensagemSMSEnum::getAll(),
            'filtro' => $request->filtro_servidor,
            'secretarias' => UnidadeOrganizacional::secretarias()->orderBy('nome')->get(),
        ]);
    }

    public function armazenaContatos(Request $request)
    {
        $this->validate($request, [
            'contatos' => 'required',
        ]);

        $contatos = Servidor
            ::whereIn('id', $request->contatos)
            ->naoPerturbe(false)
            ->comCelularMalaDireta()
            ->get();

        $request->session()->put('mensagens.contatos', $contatos);
        
        return redirect()->route('admin.envio-sms.escrever-mensagem-form');
    }

    public function escreverMensagemForm(Request $request)
    {
        if (! $request->session()->has('mensagens.contatos')) {
            return redirect()
                ->route('admin.envio-sms.selecao-contatos-form')
                ->with('error', 'Selecione os condôminos antes de escrever a mensagem');
        }
        return view('admin.envio-sms.escrever-mensagem');
    }

    public function armazenaMensagem(Request $request)
    {
        $this->validate($request, [
            'mensagem' => 'required',
        ]);

        $request->session()->put('mensagens.mensagem', $request->mensagem);

        return redirect()->route('admin.envio-sms.envia-form');
    }

    public function enviaForm(Request $request)
    {
        if (
            !$request->session()->has('mensagens.contatos') ||
            !$request->session()->has('mensagens.mensagem')
        ) {
            return redirect()->route('admin.envio-sms.selecao-contatos-form')
                ->withErrors(['Contatos devem ser selecionados e mensagem redigida']);
        }

        $contatos = $request->session()->get('mensagens.contatos');
        $mensagem = $request->session()->get('mensagens.mensagem');
        
        return view('admin.envio-sms.envia', ['contatos' => $contatos, 'mensagem' => $mensagem]);
    }

    public function envia(Request $request)
    {
        $contatos = $request->session()->get('mensagens.contatos');
        $mensagem = $request->session()->get('mensagens.mensagem');
        $tipoServidor = $request->session()->get('mensagens.tipoServidor');

        $malaDireta = new MalaDireta();
        $malaDireta->mensagem = $mensagem;
        $malaDireta->qtde_contatos = $contatos->count();
        $malaDireta->tipo_servidor = $tipoServidor->getIndex();
        $malaDireta->save();

        Notification::send($contatos, new MalaDiretaNotification($mensagem));

        $request->session()->forget('mensagens.contatos');
        $request->session()->forget('mensagens.mensagem');
        $request->session()->forget('mensagens.tipoServidor');

        return redirect()->route('admin.envio-sms.mensagens-enviadas')
            ->with('status', 'Mala direta enviada com sucesso!');
    }
}
