<?php

namespace App\Http\Controllers\Admin;

use App\Enums\StatusAgendamentoAtendimentoEnum;
use App\Enums\TipoPrestadorServicoEnum;
use App\Http\Controllers\Controller;
use App\Models\AgendamentoAtendimento;
use App\Models\AtendimentoOdontologico;
use App\Models\PrestadorServico;
use Illuminate\Http\Request;

class AtendimentoOdontologicoController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $termo = $request->busca;
        $prestadores = PrestadorServico::deTipo(TipoPrestadorServicoEnum::dentista())->get();

        if ($request->prestador_servico) {
            $prestador = PrestadorServico::find($request->prestador_servico);
            $atendimentos = AtendimentoOdontologico::doPrestador($prestador)
                                                    ->buscaPorTermo($termo)
                                                    ->orderBy('created_at', 'desc')
                                                    ->paginate();
        } else {
            $atendimentos = AtendimentoOdontologico::buscaPorTermo($termo)
                                                    ->orderBy('created_at', 'desc')
                                                    ->paginate();
        }

        

        return view('admin.atendimentos-odontologicos.index', [
            'atendimentos' => $atendimentos,
            'prestadores' => $prestadores,
            'termo' => $termo,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(AgendamentoAtendimento $agendamento)
    {
        if ($agendamento->foiAtendido()) {
            return redirect()
                    ->route('admin.atendimentos-odontologicos.edit', ['atendimento' => $agendamento->atendimento, 'agendamento' => $agendamento]);
        }

        $paciente = $agendamento->agendavel;
        $prestadores = PrestadorServico::deTipo(TipoPrestadorServicoEnum::dentista())
                                        ->comHorarioDefinido()
                                        ->get();
        return view('admin.atendimentos-odontologicos.create', [
            'agendamento' => $agendamento,
            'atendimentoAnterior' => $paciente->atendimentosOdontologicos()->latest()->first(),
            'paciente' => $paciente,
            'prestadores' => $prestadores,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(AgendamentoAtendimento $agendamento, Request $request)
    {
        $this->validate($request, [
            'tratamento_executado' => 'required',
            'prestador_servico_id' => 'required|numeric',
        ]);

        $atendimento = new AtendimentoOdontologico($request->only('tratamento_executado', 'prestador_servico_id'));
        $agendamento->status = StatusAgendamentoAtendimentoEnum::atendido();
        $agendamento->atendimento()->save($atendimento);
        $agendamento->push();

        return redirect()
                ->route('admin.agendamentos-atendimentos.index', ['tipo' => 'ODONTOLOGICO'])
                ->with('status', 'Atendimento incluído com sucesso!');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\AgendamentoAtendimento  $atendimento
     * @return \Illuminate\Http\Response
     */
    public function edit(AgendamentoAtendimento $agendamento, AtendimentoOdontologico $atendimento)
    {
        $prestadores = PrestadorServico::deTipo(TipoPrestadorServicoEnum::dentista())
            ->comHorarioDefinido()
            ->get();

        return view('admin.atendimentos-odontologicos.edit', [
            'atendimento' => $atendimento,
            'prestadores' => $prestadores,
        ]);
    }

    public function paciente(AgendamentoAtendimento $agendamento)
    {
        $paciente = $agendamento->agendavel;
        return view('admin.atendimentos-odontologicos.paciente', [
            'agendamento' => $agendamento,
            'paciente' => $paciente,
        ]);
    }

    public function historico(AgendamentoAtendimento $agendamento)
    {
        $paciente = $agendamento->agendavel;
        $atendimentos = $paciente->atendimentosOdontologicos()->orderBy('created_at', 'desc')->paginate();
        return view('admin.atendimentos-odontologicos.historico', [
            'agendamento' => $agendamento,
            'paciente' => $paciente,
            'atendimentos' => $atendimentos,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\AtendimentoOdontologico  $agendamento
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, AgendamentoAtendimento $agendamento, AtendimentoOdontologico $atendimento)
    {
        $this->validate($request, [
            'tratamento_executado' => 'required',
            'prestador_servico_id' => 'required|numeric',
        ]);

        $atendimento->fill($request->only('tratamento_executado', 'prestador_servico_id'));
        $atendimento->save();

        return redirect()
                    ->route('admin.agendamentos-atendimentos.index', ['tipo' => 'ODONTOLOGICO'])
                    ->with('status', 'Atendimento incluído com sucesso!');
    }
}
