<?php

namespace App\Http\Controllers\Admin;

use App\Enums\StatusAgendamentoAtendimentoEnum;
use App\Http\Controllers\Controller;
use App\Http\Requests\AtendimentoJuridicoStoreRequest;
use App\Models\AgendamentoAtendimento;
use App\Models\AtendimentoJuridico;
use App\Models\ProcessoJuridico;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AtendimentoJuridicoController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }
 
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(AgendamentoAtendimento $agendamento, ProcessoJuridico $processo)
    {
        if ($agendamento->foiAtendido()) {
            return redirect()
                    ->route('admin.atendimentos-juridicos.edit', ['atendimento' => $agendamento->atendimento, 'agendamento' => $agendamento]);
        }

        $paciente = $agendamento->agendavel;

        return view('admin.atendimentos-juridicos.create', [
            'agendamento' => $agendamento,
            'atendimentoAnterior' => $paciente->atendimentosOdontologicos()->latest()->first(),
            'paciente' => $paciente,
            'processo' => $processo,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(AtendimentoJuridicoStoreRequest $request, AgendamentoAtendimento $agendamento, ProcessoJuridico $processo)
    {
        DB::beginTransaction();

        $atendimento = new AtendimentoJuridico($request->only('anotacoes'));
        $atendimento->processoJuridico()->associate($processo);
        $agendamento->atendimento()->save($atendimento);
        $agendamento->status = StatusAgendamentoAtendimentoEnum::atendido();
        $agendamento->push();

        DB::commit();

        return redirect()
                ->route('admin.agendamentos-atendimentos.index', ['tipo' => 'JURIDICO'])
                ->with('status', 'Atendimento incluído com sucesso!');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\AgendamentoAtendimento  $atendimento
     * @return \Illuminate\Http\Response
     */
    public function edit(AgendamentoAtendimento $agendamento, ProcessoJuridico $processo, AtendimentoJuridico $atendimento)
    {
        return view('admin.atendimentos-juridicos.edit', [
            'atendimento' => $atendimento,
            'agendamento' => $agendamento,
            'processo' => $processo
        ]);
    }

    public function update(Request $request, AgendamentoAtendimento $agendamento, ProcessoJuridico $processo, AtendimentoJuridico $atendimento)
    {
        $this->validate($request, [
            'anotacoes' => 'required',
        ]);

        $atendimento->fill($request->only('anotacoes'));
        $atendimento->save();

        return redirect()
                    ->route('admin.processos-juridicos.edit', ['pessoa' => $processo->agendavel, 'tipo' => $processo->agendavel->tipo(), 'id' => $processo])
                    ->with('status', 'Atendimento incluído com sucesso!');
    }

    public function historico(ProcessoJuridico $processo)
    {
        $atendimentos = $processo->atendimentos()->orderBy('id', 'desc')->get();
        return view('admin.atendimentos-juridicos.historico', [
            'atendimentos' => $atendimentos,
            'processo' => $processo
        ]);
    }
}
