<?php

namespace App\Console\Commands;

use App\Models\ContaPagarReceber;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class IncluiContasRecorrentesMensaisCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'contas:inclui-recorrentes-mensais';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Inclui contas a pagar e receber recorrentes do mês corrente. É executado uma vez por mês, no seu início';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        info("Incluindo contas recorrentes mensais");
        $mesAnterior = today()->subMonth();
        $inicioMesAnterior = Carbon::createFromDate($mesAnterior->year, $mesAnterior->month, 1);
        $fimMesAnterior = $inicioMesAnterior->copy()->addMonth();

        info("Período consultado do mês anterior: {$inicioMesAnterior->format('d/m/Y')} - {$fimMesAnterior->format('d/m/Y')}");
        
        $inicioMesAtual = Carbon::createFromDate(today()->year, today()->month, 1);
        $fimMesAtual = $inicioMesAtual->copy()->addMonth();
        
        $contasRecorrentesMesAnterior = ContaPagarReceber::comRecorrenciaMensal()
                                                          ->vencemNoPeriodo($inicioMesAnterior, $fimMesAnterior)
                                                          ->get();
        info("Contas recorrentes do mês anterior", $contasRecorrentesMesAnterior->pluck('descricao', 'vencimento')->toArray());
        
        $contasMesAtual = ContaPagarReceber::vencemNoPeriodo($inicioMesAtual, $fimMesAtual)->get();
        
        info("Contas recorrentes do mês atual", $contasMesAtual->pluck('descricao', 'vencimento')->toArray());

        DB::beginTransaction();

        foreach ($contasRecorrentesMesAnterior as $conta) {
            $infoConta = $conta->only('descricao', 'vencimento', 'valor');
            info("Preparando para cópia da conta", $infoConta);

            $dataVencimentoAtualizada = $conta->vencimento->copy()->addMonth();
            debug("Data de vencimento atualizado da conta", ['conta' => $infoConta, 'dataAtualizada' => $dataVencimentoAtualizada]);

            $contaNaoExiste = $contasMesAtual->where('descricao', $conta->descricao)
                                             ->where('vencimento', $dataVencimentoAtualizada)
                                             ->count() == 0;
            if ($contaNaoExiste) {
                info("Conta recorrente não existe no mês atual. Incluindo", $infoConta);
                $contaIncluir = ContaPagarReceber::criaRecorrenteAPartirDe($conta);
                $contaIncluir->save();
                info("Conta recorrente incluída", $contaIncluir->only('descricao', 'vencimento', 'valor'));
            } else {
                info("Conta recorrente já existe no mês atual", $infoConta);
            }
        }

        DB::commit();

        info("Processo de inclusão de contas recorrentes mensais finalizado");
    }
}
