<?php

namespace App\Console\Commands;

use App\Models\ContaPagarReceber;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class IncluiContasRecorrentesAnuaisCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'contas:inclui-recorrentes-anuais';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Inclui contas a pagar e receber recorrentes do ano corrente. É executado uma vez por mês, no seu início';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        info("Incluindo contas recorrentes anuais");
        $anoAnterior = today()->subYear();
        $inicioMesAnoAnterior = Carbon::createFromDate($anoAnterior->year, $anoAnterior->month, 1);
        $fimMesAnoAnterior = $inicioMesAnoAnterior->copy()->addMonth();

        info("Período consultado do mês/ano anterior: {$inicioMesAnoAnterior->format('d/m/Y')} - {$fimMesAnoAnterior->format('d/m/Y')}");
        
        $inicioMesAtual = Carbon::createFromDate(today()->year, today()->month, 1);
        $fimMesAtual = $inicioMesAtual->copy()->addMonth();
        
        $contasRecorrentesMesAnoAnterior = ContaPagarReceber::comRecorrenciaAnual()
                                                          ->vencemNoPeriodo($inicioMesAnoAnterior, $fimMesAnoAnterior)
                                                          ->get();
        info("Contas recorrentes do mês/ano anterior", $contasRecorrentesMesAnoAnterior->pluck('descricao', 'vencimento')->toArray());
        
        $contasMesAtual = ContaPagarReceber::vencemNoPeriodo($inicioMesAtual, $fimMesAtual)->get();
        
        info("Contas recorrentes do mês/ano atual", $contasMesAtual->pluck('descricao', 'vencimento')->toArray());

        DB::beginTransaction();

        foreach ($contasRecorrentesMesAnoAnterior as $conta) {
            $infoConta = $conta->only('descricao', 'vencimento', 'valor');
            info("Preparando para cópia da conta", $infoConta);

            $dataVencimentoAtualizada = $conta->vencimento->copy()->addYear();
            debug("Data de vencimento atualizado da conta", ['conta' => $infoConta, 'dataAtualizada' => $dataVencimentoAtualizada]);

            $contaNaoExiste = $contasMesAtual->where('descricao', $conta->descricao)
                                             ->where('vencimento', $dataVencimentoAtualizada)
                                             ->count() == 0;
            if ($contaNaoExiste) {
                info("Conta recorrente não existe no mês atual. Incluindo", $infoConta);
                $contaIncluir = ContaPagarReceber::criaRecorrenteAPartirDe($conta);
                $contaIncluir->save();
                info("Conta recorrente incluída", $contaIncluir->only('descricao', 'vencimento', 'valor'));
            } else {
                info("Conta recorrente já existe no mês atual", $infoConta);
            }
        }

        DB::commit();

        info("Processo de inclusão de contas recorrentes anuais finalizado");
    }
}
