<?php

namespace Tests\Unit;

use App\Enums\DiaSemanaEnum;
use App\Enums\EspecialidadeOdontologiaEnum;
use App\Enums\TipoAgendamentoEnum;
use App\Enums\TipoPrestadorServicoEnum;
use App\Exceptions\HorarioIndisponivelFeriadoException;
use App\Models\AgendamentoAtendimento;
use App\Models\Feriado;
use App\Models\HorarioTrabalho;
use App\Models\PrestadorServico;
use App\Models\Servidor;
use App\Services\AgendaAtendimentoOdontologicoService;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase as TestCase;

class AgendaAtendimentoServiceTest extends TestCase
{
    use RefreshDatabase;

    public function test_agenda()
    {
        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $prestador = factory(PrestadorServico::class)->create([
            'tipo' => TipoPrestadorServicoEnum::dentista(),
            'especialidade' => EspecialidadeOdontologiaEnum::clinico()
        ]);
        $prestador->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::segunda()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $pessoa = factory(Servidor::class)->create();
        $tipoAgendamento = TipoAgendamentoEnum::odontologico();
        $service = new AgendaAtendimentoOdontologicoService($inicio, $fim, $prestador, $pessoa, false, false, $tipoAgendamento);

        $this->assertInstanceOf(AgendamentoAtendimento::class, $service->agenda());
    }

    public function test_agenda_feriado_prolongado()
    {
        factory(Feriado::class, 1)->create([
            'data_inicio' => '25/05/2020',
            'data_fim' => '27/05/2020'
        ]);

        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $pessoa = factory(Servidor::class)->create();
        $prestador = factory(PrestadorServico::class)->create([
            'tipo' => TipoPrestadorServicoEnum::dentista(),
            'especialidade' => EspecialidadeOdontologiaEnum::clinico()
        ]);

        $pessoa = factory(Servidor::class)->create();
        $tipoAgendamento = TipoAgendamentoEnum::odontologico();
        $service = new AgendaAtendimentoOdontologicoService($inicio, $fim, $prestador, $pessoa, false, false, $tipoAgendamento);

        $this->expectException(HorarioIndisponivelFeriadoException::class);
        $service->agenda();
    }

    public function test_eFeriado_unico_dia()
    {
        factory(Feriado::class, 1)->create([
            'data_inicio' => '25/05/2020',
            'data_fim' => '25/05/2020'
        ]);

        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $pessoa = factory(Servidor::class)->create();
        $prestador = factory(PrestadorServico::class)->create([
            'tipo' => TipoPrestadorServicoEnum::dentista(),
            'especialidade' => EspecialidadeOdontologiaEnum::clinico()
        ]);

        $pessoa = factory(Servidor::class)->create();
        $tipoAgendamento = TipoAgendamentoEnum::odontologico();
        $service = new AgendaAtendimentoOdontologicoService($inicio, $fim, $prestador, $pessoa, false, false, $tipoAgendamento);

        $this->expectException(HorarioIndisponivelFeriadoException::class);
        $service->agenda();
    }

    public function test_sobrepoeHorario_sobrepoe_antes()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 09:50');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $pessoa = factory(Servidor::class)->create();
        $prestador = factory(PrestadorServico::class)->create([
            'tipo' => TipoPrestadorServicoEnum::dentista(),
            'especialidade' => EspecialidadeOdontologiaEnum::clinico()
        ]);
        $prestador->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $tipoAgendamento = TipoAgendamentoEnum::odontologico();
        $service1 = new AgendaAtendimentoOdontologicoService($inicio1, $fim1, $prestador, $pessoa, false, false, $tipoAgendamento);
        $service2 = new AgendaAtendimentoOdontologicoService($inicio2, $fim2, $prestador, $pessoa, false, false, $tipoAgendamento);

        $this->assertFalse($service1->sobrepoeHorario());
        $agendamento = $service1->agenda();
        $this->assertInstanceOf(AgendamentoAtendimento::class, $agendamento);
        $this->assertTrue($service2->sobrepoeHorario());
    }

    public function test_sobrepoeHorario_sobrepoe_depois()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $pessoa = factory(Servidor::class)->create();
        $prestador = factory(PrestadorServico::class)->create([
            'tipo' => TipoPrestadorServicoEnum::dentista(),
            'especialidade' => EspecialidadeOdontologiaEnum::clinico()
        ]);
        $prestador->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $tipoAgendamento = TipoAgendamentoEnum::odontologico();
        $service1 = new AgendaAtendimentoOdontologicoService($inicio1, $fim1, $prestador, $pessoa, false, false, $tipoAgendamento);
        $service2 = new AgendaAtendimentoOdontologicoService($inicio2, $fim2, $prestador, $pessoa, false, false, $tipoAgendamento);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertTrue($service2->sobrepoeHorario());
    }

    public function test_sobrepoeHorario_sobrepoe_antes_depois()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 9:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $pessoa = factory(Servidor::class)->create();
        $prestador = factory(PrestadorServico::class)->create([
            'tipo' => TipoPrestadorServicoEnum::dentista(),
            'especialidade' => EspecialidadeOdontologiaEnum::clinico()
        ]);
        $prestador->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $tipoAgendamento = TipoAgendamentoEnum::odontologico();
        $service1 = new AgendaAtendimentoOdontologicoService($inicio1, $fim1, $prestador, $pessoa, false, false, $tipoAgendamento);
        $service2 = new AgendaAtendimentoOdontologicoService($inicio2, $fim2, $prestador, $pessoa, false, false, $tipoAgendamento);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertTrue($service2->sobrepoeHorario());
    }

    public function test_sobrepoeHorario_sobrepoe_entre_duas()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:15');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:30');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 12:30');

        $inicio3 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');
        $fim3 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:30');

        $pessoa1 = factory(Servidor::class)->create();
        $pessoa2 = factory(Servidor::class)->create();
        $pessoa3 = factory(Servidor::class)->create();
        $prestador = factory(PrestadorServico::class)->create([
            'tipo' => TipoPrestadorServicoEnum::dentista(),
            'especialidade' => EspecialidadeOdontologiaEnum::clinico()
        ]);
        $prestador->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $tipoAgendamento = TipoAgendamentoEnum::odontologico();
        $service1 = new AgendaAtendimentoOdontologicoService($inicio1, $fim1, $prestador, $pessoa1, false, false, $tipoAgendamento);
        $service2 = new AgendaAtendimentoOdontologicoService($inicio2, $fim2, $prestador, $pessoa2, false, false, $tipoAgendamento);
        $service3 = new AgendaAtendimentoOdontologicoService($inicio3, $fim3, $prestador, $pessoa3, false, false, $tipoAgendamento);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertFalse($service2->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service2->agenda());
        $this->assertTrue($service3->sobrepoeHorario());
    }

    public function test_sobrepoeHorario_sobrepoe_horario_exato()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $pessoa = factory(Servidor::class)->create();
        $prestador = factory(PrestadorServico::class)->create([
            'tipo' => TipoPrestadorServicoEnum::dentista(),
            'especialidade' => EspecialidadeOdontologiaEnum::clinico()
        ]);
        $prestador->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $tipoAgendamento = TipoAgendamentoEnum::odontologico();
        $service1 = new AgendaAtendimentoOdontologicoService($inicio1, $fim1, $prestador, $pessoa, false, false, $tipoAgendamento);
        $service2 = new AgendaAtendimentoOdontologicoService($inicio2, $fim2, $prestador, $pessoa, false, false, $tipoAgendamento);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertTrue($service2->sobrepoeHorario());
    }

    public function test_sobrepoeHorario_nao_sobrepoe()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:30');

        $pessoa = factory(Servidor::class)->create();
        $prestador = factory(PrestadorServico::class)->create([
            'tipo' => TipoPrestadorServicoEnum::dentista(),
            'especialidade' => EspecialidadeOdontologiaEnum::clinico()
        ]);
        $prestador->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $tipoAgendamento = TipoAgendamentoEnum::odontologico();
        $service1 = new AgendaAtendimentoOdontologicoService($inicio1, $fim1, $prestador, $pessoa, false, false, $tipoAgendamento);
        $service2 = new AgendaAtendimentoOdontologicoService($inicio2, $fim2, $prestador, $pessoa, false, false, $tipoAgendamento);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertFalse($service2->sobrepoeHorario());
    }

    public function test_verificaDisponibilidade_disponivel()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:30');

        $pessoa = factory(Servidor::class)->create();
        $prestador = factory(PrestadorServico::class)->create([
            'tipo' => TipoPrestadorServicoEnum::dentista(),
            'especialidade' => EspecialidadeOdontologiaEnum::clinico()
        ]);
        $prestador->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $tipoAgendamento = TipoAgendamentoEnum::odontologico();
        $service1 = new AgendaAtendimentoOdontologicoService($inicio1, $fim1, $prestador, $pessoa, false, false, $tipoAgendamento);
        $service2 = new AgendaAtendimentoOdontologicoService($inicio2, $fim2, $prestador, $pessoa, false, false, $tipoAgendamento);

        $service1->verificaDisponibilidade();
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $service2->verificaDisponibilidade();
    }
}
