<?php

namespace Tests\Unit;

use App\Enums\CategoriaConfiguracaoEnum;
use App\Enums\DiaSemanaEnum;
use App\Enums\StatusAgendamentoAtendimentoEnum;
use App\Enums\TipoAgendamentoEnum;
use App\Exceptions\HorarioIndisponivelFeriadoException;
use App\Exceptions\HorarioIndisponivelForaHorarioServicoException;
use App\Exceptions\ServicoAtendimentoSemHorariosDefinidosException;
use App\Exceptions\UltimoAtendimentoMuitoProximoException;
use App\Models\AgendamentoAtendimento;
use App\Models\Configuracao;
use App\Models\Dependente;
use App\Models\Feriado;
use App\Models\HorarioTrabalho;
use App\Models\ServicoAtendimento;
use App\Models\Servidor;
use App\Services\AgendaAtendimentoJuridicoService;
use Carbon\Carbon;
use ConfiguracaoSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase as TestCase;

class AgendaAtendimentoJuridicoServiceTest extends TestCase
{
    use RefreshDatabase;

    public function setUp(): void
    {
        parent::setUp();

        $this->seed(ConfiguracaoSeeder::class);
    }

    public function test_agenda()
    {
        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::segunda()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $pessoa = factory(Servidor::class)->create();
        $service = new AgendaAtendimentoJuridicoService($inicio, $fim, $pessoa, $servico);
        $agendamento = $service->agenda();

        $this->assertInstanceOf(AgendamentoAtendimento::class, $agendamento);
        $this->assertNull($agendamento->prestador_servico_id);
        $this->assertNull($agendamento->emergencia);
        $this->assertNull($agendamento->em_tratamento);
        $this->assertEquals($servico->id, $agendamento->servico_atendimento_id);
    }

    public function test_agenda_horario_indisponivel()
    {
        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::terca()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $pessoa = factory(Servidor::class)->create();
        $service = new AgendaAtendimentoJuridicoService($inicio, $fim, $pessoa, $servico);

        $this->expectException(HorarioIndisponivelForaHorarioServicoException::class);

        $service->agenda();
    }

    public function test_dentroHorarioPrestador()
    {
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $pessoa = factory(Servidor::class)->create();
        $service = new AgendaAtendimentoJuridicoService($inicio, $fim, $pessoa, $servico);

        $this->assertTrue($service->dentroHorarioServicoAtendimento());
    }

    public function test_dentroHorarioPrestador_prestador_sem_horarios()
    {
        $servico = factory(ServicoAtendimento::class)->create();
        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $pessoa = factory(Servidor::class)->create();
        $service = new AgendaAtendimentoJuridicoService($inicio, $fim, $pessoa, $servico);

        $this->expectException(ServicoAtendimentoSemHorariosDefinidosException::class);
        $service->dentroHorarioServicoAtendimento();
    }

    public function test_agenda_feriado_prolongado()
    {
        factory(Feriado::class, 1)->create([
            'data_inicio' => '25/05/2020',
            'data_fim' => '27/05/2020'
        ]);

        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $service = new AgendaAtendimentoJuridicoService($inicio, $fim, $pessoa, $servico);

        $this->expectException(HorarioIndisponivelFeriadoException::class);
        $service->agenda();
    }

    public function test_eFeriado_unico_dia()
    {
        factory(Feriado::class, 1)->create([
            'data_inicio' => '25/05/2020',
            'data_fim' => '25/05/2020'
        ]);

        $inicio = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $tipoAgendamento = TipoAgendamentoEnum::juridico();
        $service = new AgendaAtendimentoJuridicoService($inicio, $fim, $pessoa, $servico);

        $this->expectException(HorarioIndisponivelFeriadoException::class);
        $service->agenda();
    }

    public function test_sobrepoeHorario_sobrepoe_antes()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 09:50');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertTrue($service2->sobrepoeHorario());
    }

    public function test_sobrepoeHorario_sobrepoe_depois()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $tipoAgendamento = TipoAgendamentoEnum::juridico();
        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertTrue($service2->sobrepoeHorario());
    }

    public function test_sobrepoeHorario_sobrepoe_antes_depois()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 9:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertTrue($service2->sobrepoeHorario());
    }

    public function test_sobrepoeHorario_sobrepoe_entre_duas()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:15');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:30');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 12:30');

        $inicio3 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');
        $fim3 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:30');

        $pessoa1 = factory(Servidor::class)->create();
        $pessoa2 = factory(Servidor::class)->create();
        $pessoa3 = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa1, $servico);
        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa2, $servico);
        $service3 = new AgendaAtendimentoJuridicoService($inicio3, $fim3, $pessoa3, $servico);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertFalse($service2->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service2->agenda());
        $this->assertTrue($service3->sobrepoeHorario());
    }

    public function test_sobrepoeHorario_sobrepoe_horario_exato()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:30');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertTrue($service2->sobrepoeHorario());
    }

    public function test_sobrepoeHorario_nao_sobrepoe()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:30');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);

        $this->assertFalse($service1->sobrepoeHorario());
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $this->assertFalse($service2->sobrepoeHorario());
    }

    public function test_verificaDisponibilidade()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:30');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);

        $service1->verificaDisponibilidade();
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());
        $service2->verificaDisponibilidade();
    }

    public function test_agendamento_juridico_muito_proximo()
    {
        // Considerando dias mínimos de 90 dias - padrão da seed
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '22/06/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '22/06/2020 10:30');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());

        $this->expectException(UltimoAtendimentoMuitoProximoException::class);
        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);
        $service2->agenda();
    }

    public function test_agendamento_juridico_muito_proximo_permite_agendamento()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '22/06/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '22/06/2020 10:30');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $permiteAgendamentoProximo = true;

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());

        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico, false, '', $permiteAgendamentoProximo);
        $service2->agenda();
    }

    public function test_agendamento_apos_quantidade_minima_de_dias_configurada()
    {
        // Considerando dias mínimos de 90 dias - padrão da seed

        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');
        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '21/12/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '21/12/2020 11:00');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service1->agenda());

        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);
        $agendamento2 = $service2->agenda();
        $this->assertInstanceOf(AgendamentoAtendimento::class, $agendamento2);
        $this->assertTrue($agendamento2->status->isAberto());
    }

    /**
     * Titular possui agendamento e dependente tenta agendar próximo
     */
    public function test_agendamento_dependente_com_agendamento_muito_proximo_de_titular()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '04/10/2021 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '04/10/2021 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '11/10/2021 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '11/10/2021 11:00');

        $servidor = factory(Servidor::class)->create();
        $dependente = factory(Dependente::class)->create([
            'dependivel_id' => $servidor,
            'dependivel_type' => Servidor::class
        ]);
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $servidor, $servico);
        $service1->agenda();

        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $dependente, $servico);

        $this->expectException(UltimoAtendimentoMuitoProximoException::class);
        $service2->agenda();
    }

    /**
     * Dependente possui agendamento e titular tenta agendar próximo
     */
    public function test_agendamento_titular_com_agendamento_muito_proximo_de_dependente()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '04/10/2021 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '04/10/2021 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '11/10/2021 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '11/10/2021 11:00');

        $servidor = factory(Servidor::class)->create();
        $dependente = factory(Dependente::class)->create([
            'dependivel_id' => $servidor,
            'dependivel_type' => Servidor::class
        ]);
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $dependente, $servico);
        $service1->agenda();

        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $servidor, $servico);

        $this->expectException(UltimoAtendimentoMuitoProximoException::class);
        $service2->agenda();
    }

    /**
     * Dependente possui agendamento e outro dependente tenta agendar próximo
     */
    public function test_agendamento_dependente_com_agendamento_muito_proximo_de_outro_dependente()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '04/10/2021 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '04/10/2021 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '11/10/2021 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '11/10/2021 11:00');

        $servidor = factory(Servidor::class)->create();
        $dependente1 = factory(Dependente::class)->create([
            'dependivel_id' => $servidor,
            'dependivel_type' => Servidor::class
        ]);
        $dependente2 = factory(Dependente::class)->create([
            'dependivel_id' => $servidor,
            'dependivel_type' => Servidor::class
        ]);
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $dependente1, $servico);
        $service1->agenda();

        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $dependente2, $servico);

        $this->expectException(UltimoAtendimentoMuitoProximoException::class);
        $service2->agenda();
    }

    /**
     * Dependente possui agendamento e outro dependente tenta agendar após quantidade minima de dias configurada
     */
    public function test_agendamento_dependente_com_agendamento_apos_quantidade_minima_de_dias_de_outro_dependente()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '04/10/2021 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '04/10/2021 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '04/04/2022 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '04/04/2022 11:00');

        $servidor = factory(Servidor::class)->create();
        $dependente1 = factory(Dependente::class)->create([
            'dependivel_id' => $servidor,
            'dependivel_type' => Servidor::class
        ]);
        $dependente2 = factory(Dependente::class)->create([
            'dependivel_id' => $servidor,
            'dependivel_type' => Servidor::class
        ]);
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $dependente1, $servico);
        $service1->agenda();

        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $dependente2, $servico);
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service2->agenda());
    }

    /**
     * Dependente possui agendamento desmarcado e outro dependente tenta agendar próximo
     */
    public function test_agendamento_dependente_com_agendamento_desmarcado_proximo_de_outro_dependente()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '04/10/2021 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '04/10/2021 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '11/10/2021 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '11/10/2021 11:00');

        $servidor = factory(Servidor::class)->create();
        $dependente1 = factory(Dependente::class)->create([
            'dependivel_id' => $servidor,
            'dependivel_type' => Servidor::class
        ]);
        $dependente2 = factory(Dependente::class)->create([
            'dependivel_id' => $servidor,
            'dependivel_type' => Servidor::class
        ]);
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );
        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $dependente1, $servico);
        $agendamento = $service1->agenda();
        $agendamento->status = StatusAgendamentoAtendimentoEnum::desmarcou();
        $agendamento->save();

        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $dependente2, $servico);
        $this->assertInstanceOf(AgendamentoAtendimento::class, $service2->agenda());
    }

    public function test_eAMenosQueQuantidadeMinimaDeDiasDo()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '12/12/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '12/12/2020 11:00');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $agendamento1 = $service1->agenda();

        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);
        $this->assertFalse($service2->eAMenosQueQuantidadeMinimaDeDiasDo($agendamento1));
    }

    public function test_eAMenosQueQuantidadeMinimaDeDiasDo_true()
    {
        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '01/06/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '01/06/2020 11:00');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $agendamento1 = $service1->agenda();

        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);
        $this->assertTrue($service2->eAMenosQueQuantidadeMinimaDeDiasDo($agendamento1));
    }

    public function test_eAMenosQueQuantidadeMinimaDeDiasDo_desativado()
    {
        $configuracao = Configuracao::where('nome', 'DIAS_ENTRE_AGENDAMENTOS')->where('categoria', CategoriaConfiguracaoEnum::agendamentoJuridico()->getName())->first();
        $configuracao->valor = 0;
        $configuracao->save();

        $inicio1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 10:00');
        $fim1 = Carbon::createFromFormat('d/m/Y H:i', '25/05/2020 11:00');

        $inicio2 = Carbon::createFromFormat('d/m/Y H:i', '01/06/2020 10:00');
        $fim2 = Carbon::createFromFormat('d/m/Y H:i', '01/06/2020 11:00');

        $pessoa = factory(Servidor::class)->create();
        $servico = factory(ServicoAtendimento::class)->create();
        $servico->horariosTrabalho()->save(
            new HorarioTrabalho(['dia_semana' => DiaSemanaEnum::SEGUNDA()->getName(), 'inicio' => '08:00', 'fim' => '18:00']),
        );

        $service1 = new AgendaAtendimentoJuridicoService($inicio1, $fim1, $pessoa, $servico);
        $agendamento1 = $service1->agenda();

        $service2 = new AgendaAtendimentoJuridicoService($inicio2, $fim2, $pessoa, $servico);
        $this->assertFalse($service2->eAMenosQueQuantidadeMinimaDeDiasDo($agendamento1));
    }
}
