<?php

namespace App\Services;

use App\Enums\MotivoDesfiliacaoDefinitivaEnum;
use App\Enums\TipoDesligamentoSindicatoEnum;
use App\Events\ServidorDesfiliadoDefinitivamente;
use App\Events\ServidorDesfiliadoTemporariamente;
use App\Exceptions\ValidationException;
use App\Models\Servidor;
use Spatie\Enum\Enum;

class DesfiliaSindicatoService
{
    protected $servidor;
    protected $motivo;
    protected $data;
    protected $tipo;
    protected $dataPublicacaoPortaria;
    protected $dataFalecimento;
    protected $mesAnoParouDesconto;

    public function __construct(
        Servidor $servidor,
        Enum $motivo,
        TipoDesligamentoSindicatoEnum $tipo,
        \DateTime $data,
        \DateTime $dataPublicacaoPortaria = null,
        \DateTime $dataFalecimento = null,
        string $mesAnoParouDesconto = null
    ) {
        $this->servidor = $servidor;
        $this->motivo = $motivo;
        $this->data = $data;
        $this->tipo = $tipo;
        $this->dataPublicacaoPortaria = $dataPublicacaoPortaria;
        $this->dataFalecimento = $dataFalecimento;
        $this->mesAnoParouDesconto = $mesAnoParouDesconto;
    }

    public function desfiliaDefinitivamente()
    {
        if ($this->servidor->desfiliadoTemporariamente()) {
            $matricula = $this->servidor->matriculas()->desligadaTemporariamente()->first();
        } else {
            $matricula = $this->servidor->matriculas()->ativa()->first();
        }

        $matricula->data_desligamento = $this->data;
        $matricula->motivo_desligamento = $this->motivo->getIndex();
        $matricula->tipo_desligamento = TipoDesligamentoSindicatoEnum::definitivo();
        $matricula->data_religamento_extraordinario = null;
        $matricula->religamento_extraordinario = false;
        $matricula->motivo_religamento_extraordinario = null;

        if ($this->motivo->isFalecido()) {
            $this->servidor->data_falecimento = $this->dataFalecimento;
            $this->servidor->falecido = true;
        } else {
            $motivos = [
                MotivoDesfiliacaoDefinitivaEnum::demitido(),
                MotivoDesfiliacaoDefinitivaEnum::exoneracao(),
                MotivoDesfiliacaoDefinitivaEnum::terminoContrato()
            ];

            if ($this->motivo->isAny($motivos)) {
                $matricula->data_publicacao_portaria = $this->dataPublicacaoPortaria;
            }
        }


        $this->servidor->save();
        $matricula->save();

        event(new ServidorDesfiliadoDefinitivamente($this->servidor, $matricula));

        return $matricula;
    }

    public function desfilia()
    {
        $this->valida();

        if ($this->tipo->isDefinitivo()) {
            return $this->desfiliaDefinitivamente();
        }

        return $this->desfiliaTemporariamente();
    }

    function desfiliaTemporariamente()
    {
        $matricula = $this->servidor->matriculas()->ativa()->first();

        $matricula->data_desligamento = $this->data;
        $matricula->motivo_desligamento = $this->motivo->getIndex();
        $matricula->tipo_desligamento = $this->tipo;
        $matricula->data_religamento_extraordinario = null;
        $matricula->religamento_extraordinario = false;
        $matricula->motivo_religamento_extraordinario = null;
        $matricula->mes_ano_parou_desconto = $this->mesAnoParouDesconto;
        $matricula->save();

        event(new ServidorDesfiliadoTemporariamente($this->servidor, $matricula));

        return $matricula;
    }

    public function valida()
    {
        $matriculaAtiva = $this->servidor->matriculas()->ativa()->first();
        $desfiliadoTemporariamente = $this->servidor->desfiliadoTemporariamente();

        if (!$matriculaAtiva && !$desfiliadoTemporariamente) {
            throw new ValidationException('Servidor deve estar filiado ao sindicato ou com sua matrícula desligada temporariamente');
        }

        return true;
    }
}
