<?php

namespace App\Models;

use App\Enums\ProventoAposentadoEnum;
use App\Enums\RegimeServidorEnum;
use App\Enums\TipoAposentadoriaEnum;
use App\Traits\TemEndereco;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class RegistroServidor extends Model implements AuditableContract
{
    use TemEndereco, Auditable;

    protected $table = 'registros_servidores';
    protected $guarded = ['id', 'created_at', 'updated_at'];
    protected $appends = ['ativo'];

    public function servidor()
    {
        return $this->belongsTo(Servidor::class);
    }

    public function cargo()
    {
        return $this->belongsTo(Cargo::class);
    }

    public function cargoReadaptado()
    {
        return $this->belongsTo(Cargo::class, 'cargoreadaptado_id');
    }

    public function unidadeOrganizacional()
    {
        return $this->belongsTo(UnidadeOrganizacional::class, 'unidadeorganizacional_id');
    }

    public function nivelSalarial()
    {
        return $this->belongsTo(NivelSalarial::class);
    }

    public function scopeAtivos($query)
    {
        return $query->whereNull('data_desligamento');
    }

    public function scopeComRegime($query, RegimeServidorEnum $regime)
    {
        return $query->where('regime', $regime->getIndex());
    }

    public function getDataAdmissaoAttribute()
    {
        if (!$this->attributes['data_admissao']) {
            return;
        }

        if ($this->attributes['data_admissao'] instanceof Carbon) {
            return $this->attributes['data_admissao'];
        }

        return Carbon::create($this->attributes['data_admissao']);
    }

    public function setDataAdmissaoAttribute($data)
    {
        if ($data) {
            $this->attributes['data_admissao'] = Carbon::createFromFormat('d/m/Y', $data);
        }
    }

    public function getDataAposentadoriaAttribute()
    {
        if (!$this->attributes['data_aposentadoria']) {
            return;
        }

        if ($this->attributes['data_aposentadoria'] instanceof Carbon) {
            return $this->attributes['data_aposentadoria'];
        }

        return Carbon::create($this->attributes['data_aposentadoria']);
    }

    public function setDataAposentadoriaAttribute($data)
    {
        if ($data) {
            $this->attributes['data_aposentadoria'] = Carbon::createFromFormat('d/m/Y', $data);
        }
    }

    public function getRegimeAttribute()
    {
        return RegimeServidorEnum::make($this->attributes['regime']);
    }

    public function setRegimeAttribute(RegimeServidorEnum $regime = null)
    {
        $this->attributes['regime'] = $regime->getIndex();
    }

    public function getProventosAttribute()
    {
        if ($this->attributes['proventos']) {
            return ProventoAposentadoEnum::make($this->attributes['proventos']);
        }

        return null;
    }

    public function setProventosAttribute(ProventoAposentadoEnum $proventos = null)
    {
        if ($proventos) {
            $this->attributes['proventos'] = $proventos->getIndex();
        }
    }

    public function getTipoAposentadoriaAttribute()
    {
        if ($this->attributes['tipo_aposentadoria']) {
            return TipoAposentadoriaEnum::make($this->attributes['tipo_aposentadoria']);
        }

        return null;
    }

    public function setTipoAposentadoriaAttribute(TipoAposentadoriaEnum $tipoAposentadoria = null)
    {
        if ($tipoAposentadoria) {
            $this->attributes['tipo_aposentadoria'] = $tipoAposentadoria->getIndex();
        }
    }

    public function getDataDesligamentoAttribute()
    {
        if (!$this->attributes['data_desligamento']) {
            return;
        }

        if ($this->attributes['data_desligamento'] instanceof Carbon) {
            return $this->attributes['data_desligamento'];
        }

        if ($this->attributes['data_desligamento']) {
            return Carbon::create($this->attributes['data_desligamento']);
        }

        return null;
    }

    public function setDataDesligamentoAttribute($data)
    {
        if ($data) {
            $this->attributes['data_desligamento'] = Carbon::createFromFormat('d/m/Y', $data);
        }
    }

    public function getAtivoAttribute()
    {
        return $this->attributes['data_desligamento'] == null;
    }
}
