<?php

namespace App\Models;

use App\Enums\EspecialidadeOdontologiaEnum;
use App\Enums\SexoEnum;
use App\Enums\TipoPrestadorServicoEnum;
use App\Traits\Contatavel;
use App\Traits\TemProcessoJuridico;
use Illuminate\Database\Eloquent\Model;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class PrestadorServico extends Model implements AuditableContract
{
    use Auditable;
    use Contatavel;
    use TemProcessoJuridico;
    
    protected $table = 'prestadores_servicos';
    protected $guarded = ['id', 'created_at', 'updated_at', 'deleted_at'];

    public function endereco()
    {
        return $this->morphOne(Endereco::class, 'enderecavel')->withDefault();
    }

    public function agendamentos()
    {
        return $this->hasMany(AgendamentoAtendimento::class);
    }

    public function horariosTrabalho()
    {
        return $this->morphMany(HorarioTrabalho::class, 'trabalhavel');
    }

    public function usuario()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function temUsuario()
    {
        return $this->usuario != null;
    }

    public function scopeDeTipo($query, TipoPrestadorServicoEnum $tipo)
    {
        return $query->where('tipo', $tipo->getIndex());
    }

    public function scopeComHorarioDefinido($query)
    {
        return $query->has('horariosTrabalho');
    }

    public function scopeBuscaPorTermo($query, $termo)
    {
        if ($termo) {
            $query->where('nome', 'ilike', "%{$termo}%")
                ->orWhere('funcao', 'ilike', "%{$termo}%");
        }

        return $query->orderBy('nome');
    }

    public function getSexoAttribute()
    {
        if (! $this->attributes['sexo']) {
            return;
        }

        return SexoEnum::make($this->attributes['sexo']);
    }

    public function setSexoAttribute(SexoEnum $sexo = null)
    {
        if (! $sexo) {
            $this->attributes['sexo'] = null;
            return;
        }

        return $this->attributes['sexo'] = $sexo->getIndex();
    }

    public function getTipoAttribute()
    {
        if (!$this->attributes['tipo']) {
            return;
        }

        return TipoPrestadorServicoEnum::make($this->attributes['tipo']);
    }

    public function setTipoAttribute(TipoPrestadorServicoEnum $tipo = null)
    {
        if (!$tipo) {
            $this->attributes['tipo'] = null;
            return;
        }

        return $this->attributes['tipo'] = $tipo->getIndex();
    }

    public function getEspecialidadeAttribute()
    {
        if (! isset($this->attributes['especialidade'])) {
            return;
        }

        return EspecialidadeOdontologiaEnum::make($this->attributes['especialidade']);
    }

    public function setEspecialidadeAttribute(EspecialidadeOdontologiaEnum $especialidade = null)
    {
        if (!$especialidade) {
            $this->attributes['especialidade'] = null;
            return;
        }

        return $this->attributes['especialidade'] = $especialidade->getIndex();
    }

    public function setCpfAttribute($cpf)
    {
        $this->attributes['cpf'] = str_replace(['.', '-'], '', $cpf);
    }

    public function setCnpjAttribute($cnpj)
    {
        $this->attributes['cnpj'] = str_replace(['.', '-', '/'], '', $cnpj);
    }
}
