<?php

namespace App\Models;

use App\Enums\SexoEnum;
use App\Traits\Atendivel;
use App\Traits\Contatavel;
use App\Traits\TemProcessoJuridico;
use Carbon\Carbon;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class Funcionario extends Pessoa implements AuditableContract
{
    use Auditable, Contatavel, Atendivel, TemProcessoJuridico;

    protected $guarded = ['id', 'created_at', 'updated_at', 'deleted_at'];

    public function tipo()
    {
        return 'funcionario';
    }

    public function tipoFormatado()
    {
        return 'funcionário';
    }
    
    public function endereco()
    {
        return $this->morphOne(Endereco::class, 'enderecavel')->withDefault();
    }

    public function dependentes()
    {
        return $this->morphMany(Dependente::class, 'dependivel');
    }

    public function convenios()
    {
        return $this->morphMany(ConvenioPessoa::class, 'conveniavel');
    }

    public function usuario()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function temUsuario()
    {
        return $this->usuario != null;
    }

    public function scopeBuscaPorTermo($query, $termo)
    {
        if ($termo) {
            $query->where('nome', 'ilike', "%{$termo}%")
                ->orWhere('funcao', 'ilike', "%{$termo}%");
        }

        return $query->orderBy('nome');
    }

    public function getSexoAttribute()
    {
        if (!$this->attributes['sexo']) {
            return;
        }

        return SexoEnum::make($this->attributes['sexo']);
    }

    public function setSexoAttribute(SexoEnum $sexo = null)
    {
        if (!$sexo) {
            $this->attributes['sexo'] = null;
            return;
        }

        return $this->attributes['sexo'] = $sexo->getIndex();
    }

    public function setCpfAttribute($cpf)
    {
        $this->attributes['cpf'] = str_replace(['.', '-'], '', $cpf);
    }

    public function setDataNascimentoAttribute($data)
    {
        $this->attributes['data_nascimento'] = Carbon::createFromFormat('d/m/Y', $data);
    }

    public function getDataNascimentoAttribute()
    {
        return Carbon::create($this->attributes['data_nascimento']);
    }

    public function routeNotificationForTwilio($notification)
    {
        if ($this->celularMalaDireta()) {
            return $this->celularMalaDireta()->getNumero();
        }

        return null;
    }
}
