<?php

namespace App\Models;

use App\Enums\FormaPagamentoContaPagarEnum;
use App\Enums\FormaPagamentoContaReceberEnum;
use App\Enums\RecorrenciaContaPagaReceberEnum;
use App\Enums\StatusContaPagarEnum;
use App\Enums\TipoContaPagarReceberEnum;
use App\Enums\TipoDespesaContaPagarReceberEnum;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class ContaPagarReceber extends Model implements AuditableContract
{
    use Auditable;

    protected $table = 'contas_pagar_receber';
    protected $guarded = ['created_at', 'updated_at'];

    /**
     * Copia conta a pagar/receber para fins de criação de conta recorrente
     *
     */
    public static function criaRecorrenteAPartirDe(ContaPagarReceber $conta)
    {
        $contaIncluir = new self();
        $contaIncluir->descricao = $conta->descricao;
        $contaIncluir->attributes['valor'] = $conta->valor;
        $contaIncluir->status = StatusContaPagarEnum::aberto();
        $contaIncluir->recorrencia = $conta->recorrencia;
        $contaIncluir->tipo = $conta->tipo;
        $contaIncluir->tipo_despesa = $conta->tipo_despesa;
        $contaIncluir->responsavel_despesa = $conta->responsavel_despesa;
        $contaIncluir->observacoes = $conta->observacoes;
        $contaIncluir->classificacao_conta_id = $conta->classificacao_conta_id;

        if ($conta->recorrencia->isMensal()) {
            $vencimentoAtualizado = $conta->vencimento->copy()->addMonth();
        } else {
            $vencimentoAtualizado = $conta->vencimento->copy()->addYear();
        }

        $contaIncluir->vencimento = $vencimentoAtualizado;
        return $contaIncluir;
    }

    public function scopeSearch($query, $busca, StatusContaPagarEnum $status, $vencimento = [])
    {
        if ($status->isVencido($status)) {
            $query->where('vencimento', '<', Carbon::today()->setTime(0, 0, 0))
                ->where('status', StatusContaPagarEnum::aberto()->getIndex());
        } elseif ($status->isAberto()) {
            $query->where(function ($query) use ($status) {
                $query->where('status', $status->getIndex())
                    ->where('vencimento', '>=', Carbon::today()->setTime(0, 0, 0));
            });
        }

        if ($vencimento) {
            $query->where(function ($query) use ($vencimento) {
                $query->whereBetween('vencimento', $vencimento);
            });
        }

        if ($busca) {
            $query->where(function ($query) use ($busca) {
                $query->where('descricao', 'ilike', "%{$busca}%")
                    ->orWhere('observacoes', 'ilike', "%{$busca}%");
            });
        }



        return $query
            ->orderBy('vencimento', 'asc')
            ->orderBy('descricao', 'asc');
    }

    public function scopeMovimentos($query, $periodo)
    {
        $query->where(function ($query) use ($periodo) {
            $query->whereBetween('data_pagamento', $periodo);
        })->where('status', StatusContaPagarEnum::pago()->getIndex());

        return $query
            ->orderBy('data_pagamento', 'asc')
            ->orderBy('descricao', 'asc');
    }

    public function scopeComNotaFiscal($query, $notaFiscal = null)
    {
        return $query->when($notaFiscal, function ($query, $notaFiscal) {
            return $query->where('nota_fiscal', $notaFiscal);
        });
    }

    public function scopeComRecorrenciaMensal($query)
    {
        return $query->where('recorrencia', RecorrenciaContaPagaReceberEnum::mensal()->getIndex());
    }

    public function scopeComRecorrenciaAnual($query)
    {
        return $query->where('recorrencia', RecorrenciaContaPagaReceberEnum::anual()->getIndex());
    }

    public function scopeDebitos($query)
    {
        return $query->where('tipo', TipoContaPagarReceberEnum::saida()->getIndex());
    }

    public function scopeEntradas($query)
    {
        return $query->where('tipo', TipoContaPagarReceberEnum::entrada()->getIndex());
    }

    public function scopeSaidas($query)
    {
        return $query->where('status', StatusContaPagarEnum::pago()->getIndex());
    }

    public function scopeVencemNoPeriodo($query, Carbon $inicio, Carbon $fim)
    {
        return $query->whereBetween('vencimento', [$inicio, $fim]);
    }

    public function scopePagasNoPeriodo($query, $periodo)
    {
        return $query->whereBetween('data_pagamento', $periodo);
    }

    public static function saldo($periodo)
    {
        $entrada = self::entradas()->pagasNoPeriodo($periodo)->get()->reduce(function ($total, $entrada) {
            return $total + (float) $entrada->valor_pago;
        });
        $saida = self::saidas()->pagasNoPeriodo($periodo)->get()->reduce(function ($total, $saida) {
            return $total + (float) $saida->valor_pago;
        });

        return (object) [
            'saldo' => $entrada - $saida,
            'entrada' => $entrada,
            'saida' => $saida,
        ];
    }

    public function classificacao()
    {
        return $this->belongsTo(ClassificacaoConta::class, 'classificacao_conta_id');
    }

    public function foiPaga()
    {
        return $this->getOriginal('valor_pago') != null;
    }

    public function setValorAttribute($valor)
    {
        $this->attributes['valor'] = str_replace(',', '.', str_replace('.', '', $valor));
    }

    public function setValorPagoAttribute($valor)
    {
        if ($valor) {
            $this->attributes['valor_pago'] = str_replace(',', '.', str_replace('.', '', $valor));
        } else {
            $this->attributes['valor_pago'] = 0;
        }
    }

    public function getDataPagamentoAttribute()
    {
        if ($this->attributes['data_pagamento']) {
            return Carbon::create($this->attributes['data_pagamento']);
        }

        return null;
    }

    public function setDataPagamentoAttribute($dataPagamento)
    {
        if ($dataPagamento) {
            $this->attributes['data_pagamento'] = Carbon::createFromFormat('d/m/Y', $dataPagamento);
        }
    }

    public function getVencimentoAttribute()
    {
        if ($this->attributes['vencimento'] instanceof Carbon) {
            return $this->attributes['vencimento'];
        }

        return Carbon::create($this->attributes['vencimento']);
    }

    public function setVencimentoAttribute($vencimento)
    {
        if ($vencimento instanceof Carbon) {
            $this->attributes['vencimento'] = $vencimento;
        } else {
            $this->attributes['vencimento'] = Carbon::createFromFormat('d/m/Y', $vencimento);
        }
    }

    public function getTipoAttribute()
    {
        if (!$this->attributes['tipo']) {
            return null;
        }

        return TipoContaPagarReceberEnum::make($this->attributes['tipo']);
    }

    public function setTipoAttribute(TipoContaPagarReceberEnum $tipo = null)
    {
        if ($tipo) {
            $this->attributes['tipo'] = $tipo->getIndex();
        }
    }

    public function getStatusAttribute()
    {
        if (!$this->attributes['status']) {
            return null;
        }

        return StatusContaPagarEnum::make($this->attributes['status']);
    }

    public function setStatusAttribute(StatusContaPagarEnum $status = null)
    {
        if ($status) {
            $this->attributes['status'] = $status->getIndex();
        }
    }

    public function getTipoDespesaAttribute()
    {
        if (!$this->attributes['tipo_despesa']) {
            return null;
        }

        return TipoDespesaContaPagarReceberEnum::make($this->attributes['tipo_despesa']);
    }

    public function setFormaPagamentoAttribute(FormaPagamentoContaPagarEnum $forma = null)
    {
        if ($forma) {
            $this->attributes['forma_pagamento'] = $forma->getIndex();
        }
    }

    public function getFormaPagamentoAttribute()
    {
        if (!$this->attributes['forma_pagamento']) {
            return null;
        }

        return FormaPagamentoContaPagarEnum::make($this->attributes['forma_pagamento']);
    }

    public function setFormaRecebimentoAttribute(FormaPagamentoContaReceberEnum $forma = null)
    {
        if ($forma) {
            $this->attributes['forma_recebimento'] = $forma->getIndex();
        }
    }

    public function getFormaRecebimentoAttribute()
    {
        if (!$this->attributes['forma_recebimento']) {
            return null;
        }

        return FormaPagamentoContaReceberEnum::make($this->attributes['forma_recebimento']);
    }

    public function setTipoDespesaAttribute(TipoDespesaContaPagarReceberEnum $tipo = null)
    {
        if ($tipo) {
            $this->attributes['tipo_despesa'] = $tipo->getIndex();
        }
    }

    public function getRecorrenciaAttribute()
    {
        if (!$this->attributes['recorrencia']) {
            return null;
        }

        return RecorrenciaContaPagaReceberEnum::make($this->attributes['recorrencia']);
    }

    public function setRecorrenciaAttribute(RecorrenciaContaPagaReceberEnum $recorrencia = null)
    {
        $this->attributes['recorrencia'] = optional($recorrencia)->getIndex() ?? null;
    }
}
