<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\URL;
use OwenIt\Auditing\Models\Audit as BaseAudit;

class Audit extends BaseAudit
{
    private const AUDIT_TYPE_READABLE = [
        ContaPagarReceber::class   => 'Conta a pagar/receber',
        Feriado::class             => 'Feriado',
        Cargo::class               => 'Cargo',
        Role::class                => 'Perfil',
        User::class                => 'Usuário',
        MalaDireta::class          => 'Mala direta',
        Servidor::class            => 'Servidor',
        PrestadorServico::class    => 'Prestador de serviço',
        Funcionario::class         => 'Funcionário',
        PerguntaAnamnese::class    => 'Pergunta da anamnese',
        PlanoConta::class          => 'Plano de contas',
        RegistroServidor::class    => 'Registro de servidor',
    ];

    private const MODEL_DESCRIPTION_FIELD = [
        ContaPagarReceber::class  => 'descricao',
        Feriado::class            => 'titulo',
        Cargo::class              => 'nome',
        Role::class               => 'name',
        User::class               => 'name',
        MalaDireta::class         => 'mensagem',
        Servidor::class           => 'nome',
        PrestadorServico::class   => 'nome',
        Funcionario::class        => 'nome',
        PerguntaAnamnese::class   => 'descricao',
        PlanoConta::class         => 'nome',
        RegistroServidor::class   => 'numero',
    ];

    private const EVENTS_READABLE = [
        'created' => 'inclusão',
        'updated' => 'edição',
        'deleted' => 'exclusão',
    ];

    public static function search($intervalo = '', $evento = '')
    {
        $query = self::query();

        if ($intervalo) {
            $datas = explode(' - ', $intervalo);
            $inicio = Carbon::createFromFormat('d/m/Y', $datas[0])->setTime(0, 0, 0);
            $fim = Carbon::createFromFormat('d/m/Y', $datas[1])->setTime(0, 0, 0);

            $query->where(function ($query) use ($inicio, $fim) {
                $query->whereBetween('created_at', [$inicio, $fim]);
            });
        }

        if ($evento) {
            $query->where(function ($query) use ($evento) {
                $query->where('event', $evento);
            });
        }

        $query->orderBy('created_at', 'desc');

        $itensPaginados = $query->paginate();
        $transformados = $itensPaginados->getCollection()->transform(function ($log) {
            $log->auditable_type_readable = self::AUDIT_TYPE_READABLE[$log->auditable_type];
            $log->event_readable = self::EVENTS_READABLE[$log->event];
            $field = isset($log->auditable) ? self::MODEL_DESCRIPTION_FIELD[get_class($log->auditable)] : '';
            $log->description = isset($log->auditable) ? $log->auditable->$field : '';
            return $log;
        });

        return new LengthAwarePaginator(
            $transformados,
            $itensPaginados->total(),
            $itensPaginados->perPage(),
            $itensPaginados->currentPage(),
            [
                'path' => URL::current(),
                'query' => [
                    'page' => $itensPaginados->currentPage()
                ]
            ]
        );
    }

    public static function find(int $id)
    {
        $log = BaseAudit::find($id);

        $log->auditable_type_readable = self::AUDIT_TYPE_READABLE[$log->auditable_type];
        $log->event_readable = self::EVENTS_READABLE[$log->event];
        $field = isset($log->auditable) ? self::MODEL_DESCRIPTION_FIELD[get_class($log->auditable)] : '';
        $log->description = isset($log->auditable) ? $log->auditable->$field : '';

        return $log;
    }
}
