<?php

namespace App\Http\Requests;

use App\Enums\RegimeServidorEnum;
use App\Models\RegistroServidor;
use App\Rules\CPFRule;
use Illuminate\Foundation\Http\FormRequest;

class ServidorStoreRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    protected function prepareForValidation(): void
    {
        $this->merge([
            'cpf' => str_replace(['.', '-'], '', $this->cpf)
        ]);
        $this->merge([
            'cep' => str_replace(['.', '-'], '', $this->cep)
        ]);
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'nome' => 'required|max:150',
            'nome_social' => 'nullable|max:150',
            'cpf' => $this->ruleCpf(),
            'rg' => 'nullable|max:15',
            'data_nascimento' => 'nullable|date_format:d/m/Y|before:today',
            'nome_pai' => 'nullable|max:150',
            'nome_mae' => 'nullable|max:150',
            'naturalidade' => 'nullable|max:150',
            'naturalidade_uf' => 'nullable|required_with:naturalidade|max:2',
            'nacionalidade' => 'nullable|max:150',
            'estado_civil' => 'nullable',
            'sexo' => 'nullable',
            'nao_perturbe' => 'required|boolean',
            'nivel_ensino' => 'nullable|numeric',
            'observacoes' => 'nullable|max:150',

            'numero_registro' => 'required|numeric',
            'regime' => 'required|numeric',
            'data_admissao' => 'nullable|date_format:d/m/Y',
            'data_desligamento' => 'nullable|date_format:d/m/Y|after_or_equal:data_admissao',
            'cargo_id' => 'nullable|numeric',
            'readaptado' => 'nullable|boolean',
            'cargoreadaptado_id' => 'required_if:readaptado,1',
            'unidadeorganizacional_id' => 'nullable|numeric',

            'paridade' => "nullable|boolean",
            'proventos' => "nullable|numeric",
            'tipo_aposentadoria' => "nullable|numeric",
            'data_aposentadoria' => "nullable|date|after:data_admissao",
            'portaria_aposentadoria' => "nullable|string|max:20",

            'cep' => 'nullable|numeric',
            'logradouro' => 'required_with:cep|max:150',
            'bairro' => 'required_with:cep|max:50',
            'cidade' => 'required_with:cep|max:100',
            'numero' => 'required_with:cep|nullable|numeric',
            'estado' => 'required_with:cep|max:2',
            'observacoes_endereco' => 'nullable|max:100',

            'telefone.*.numero' => 'nullable|regex:/^\(\d{2}\) \d{4,5}-\d{4}$/',
            'telefone.0.numero' => 'required',
            'email.*.endereco' => 'nullable|email',
        ];
    }

    private function ruleCpf()
    {
        $required = 'required';

        if ($this->input('cpf_opcional', false)) {
            $required = 'nullable';
        }

        return [$required, 'max:14', 'unique:servidores', new CPFRule];
    }

    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            if ($this->registroJaExiste()) {
                $validator->errors()->add('registro', 'Já existe um registro com o número e regime informados!');
            }
        });
    }

    private function registroJaExiste()
    {
        return RegistroServidor::where('numero', $this->numero_registro)->where('regime', $this->regime)->count() > 0;
    }

    public function attributes()
    {
        return [
            'cpf' => 'CPF',
            'rg' => 'RG',
            'numero_registro' => 'registro',
            'cep' => 'CEP',
            'nome_mae' => 'nome da mãe',
            'nome_pai' => 'nome do pai',
            'numero' => 'número',
            'data_admissao' => 'data admissão',
            'observacoes' => 'observações',
            'observacoes_endereco' => 'observações do endereço',
            'cep_lotacao' => 'CEP lotação',
            'logradouro_lotacao' => 'logradouro lotação',
            'bairro_lotacao' => 'bairro lotação',
            'cidade_lotacao' => 'cidade lotação',
            'numero_lotacao' => 'número lotação',
            'estado_lotacao' => 'estado lotação',
            'observacoes_endereco_lotacao' => 'observações endereço lotação',
            'data_aposentadoria' => 'data de aposentadoria',
            'portaria_aposentadoria' => 'portaria de aposentadoria',
            'naturalidade_uf' => 'Naturalidade/UF',
            'email.*.endereco' => 'e-mail',
        ];
    }

    public function messages()
    {
        return [
            'data_nascimento.before' => 'O campo data de nascimento deve ser uma data anterior à data atual.',
            'data_admissao_sindicato.afer' => 'O campo data adm. sindicato deve ser uma data posterior à data de nascimento.',
            'data_admissao.after' => 'O campo data de admissão deve ser uma data posterior à data de nascimento.',
            'data_desligamento.after_or_equal' => 'O campo data de desligamento deve ser uma data posterior ou igual à data de admissão.',
            'cargoreadaptado_id.required_if' => 'O campo cargo readaptado é obrigatório quando o servidor é readaptado.',
            'data_admissao_sindicato.required_if' => 'O campo Data adm. sindicato é obrigatório quando servidor for associado.',
            'motivo_filiacao.required_if' => 'O campo Motivo filiação é obrigatório quando servidor for associado.',
            'cep.required_if' => 'O endereço é obrigatório quando não se utiliza o endereço da unidade.',
            'paridade.required_if' => 'O campo Paridade é obrigatório quando o servidor for aposentado',
            'proventos.required_if' => "O campo Proventos é obrigatório quando o servidor for aposentado",
            'tipo_aposentadoria.required_if' => "O campo tipo de aposentadoria é obrigatório quando o servidor for aposentado",
            'data_aposentadoria.required_if' => "O campo data de aposentadoria é obrigatório quando o servidor for aposentado",
            'portaria_aposentadoria.required_if' => "O campo portaria é obrigatório quando o servidor for aposentado",
            'naturalidade_uf.required_with' => 'O campo Naturalidade/UF deve ser informado junto com a naturalidade',
            'telefone.regex' => 'O campo telefone deve ser um telefone válido',
        ];
    }
}
