<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Funcionario;
use App\Models\PrestadorServico;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;

class UsuariosController extends Controller
{
    public function __construct()
    {
        $this->middleware(['role:admin']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $busca = $request->busca;
        $users = User::search($busca)->comRole(['admin', 'funcionario', 'prestador'])->paginate();

        return view('admin.usuarios.index', compact('users', 'busca'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $funcionarios = Funcionario::orderBy('nome')->get();
        $prestadores = PrestadorServico::orderBy('nome')->get();
        return view('admin.usuarios.create', [
            'funcionarios' => $funcionarios,
            'prestadores' => $prestadores,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'name' => 'required',
            'email' => 'required|email|unique:users',
            'password' => 'required|confirmed|min:8',
        ]);

        $user = new User($request->all());
        $user->password = Hash::make($request->password);

        DB::beginTransaction();
        $user->save();
        
        if ($request->funcionario_id) {
            $funcionario = Funcionario::find($request->funcionario_id);
            $funcionario->usuario()->associate($user);
            $funcionario->save();
        } elseif ($request->prestador_servico_id) {
            $prestador = PrestadorServico::find($request->prestador_servico_id);
            $prestador->usuario()->associate($user);
            $prestador->save();
            $user->assignRole('prestador');
        }

        DB::commit();

        return redirect()
            ->route('admin.usuarios.index')
            ->with('status', 'Usuário incluído com sucesso!');
    }

    public function edit(User $usuario)
    {
        $funcionarios = Funcionario::orderBy('nome')->get();
        $prestadores = PrestadorServico::orderBy('nome')->get();
        return view('admin.usuarios.edit', [
            'usuario' => $usuario,
            'funcionarios' => $funcionarios,
            'prestadores' => $prestadores,
        ]);
    }

    public function update(Request $request, User $usuario)
    {
        $this->validate($request, [
            'name' => 'required',
            'email' => ['required', 'email', Rule::unique('users')->ignore($usuario)],
        ]);
        
        $usuario->fill($request->all());
        
        DB::beginTransaction();

        if ($request->funcionario_id) {
            $funcionario = Funcionario::find($request->funcionario_id);
            $funcionario->usuario()->associate($usuario);
            $usuario->funcionario()->save($funcionario);
        } elseif ($request->prestador_servico_id) {
            $prestador = PrestadorServico::find($request->prestador_servico_id);
            $prestador->usuario()->associate($usuario);
            $usuario->prestador()->save($prestador);
        }

        $usuario->push();

        DB::commit();

        return redirect()
            ->route('admin.usuarios.index')
            ->with('status', 'Usuário alterado com sucesso!');
    }

    public function destroy(User $usuario)
    {
        $quantidadeUsuarios = User::search()->count();
        
        if ($quantidadeUsuarios == 1) {
            return redirect()->back()->withErrors('Não é possível excluir esse usuário, pois é necessário ao menos um usuário administrador');
        }
        
        $usuario->delete();

        return redirect()
                ->route('admin.usuarios.index')
                ->with('status', 'Usuário excluído com sucesso!');
    }

    public function alteraSenhaForm(User $usuario)
    {
        return view('admin.usuarios.altera-senha', compact('usuario'));
    }

    public function alteraSenha(User $usuario, Request $request)
    {
        $this->validate($request, [
            'senha_nova' => 'required|confirmed|min:8',
        ]);

        $usuario->password = Hash::make($request->senha_nova);
        $usuario->save();

        Log::info("Usuário '{$usuario->email}' alterou senha de outro usuário", [
            'usuario-alterado' => $usuario,
        ]);

        return redirect()
            ->route('admin.usuarios.edit', ['usuario' => $usuario])
            ->with('status', 'Senha alterada com sucesso!');
    }
}
