<?php

namespace App\Http\Controllers\Admin\Servidor;

use App\Http\Controllers\Controller;
use App\Models\DocumentoAssociado;
use App\Models\Servidor;
use App\Models\TipoDocumento;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class DocumentoAssociadoController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Servidor $servidor)
    {
        return view('admin.servidores.documentos-associados.index', [
            'documentos' => $servidor->documentos()->paginate(),
            'servidor' => $servidor
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Servidor $servidor)
    {
        $tipos = TipoDocumento::orderBy('nome')->get();
        return view('admin.servidores.documentos-associados.create', [
            'tipos' => $tipos,
            'servidor' => $servidor
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, Servidor $servidor)
    {
        $this->validate($request, [
            'arquivo' => [
                'required',
                'file',
                implode(',', config('app.mimetypes-documentos-associados'))
            ],
            'tipo' => 'required|numeric',
            'observacoes' => 'nullable|max:100',
        ]);

        $documento = new DocumentoAssociado();
        $tipo = TipoDocumento::find($request->tipo);
        $documento->tipo()->associate($tipo);
        $documento->caminho = $request->file('arquivo')->store("documentos-associados/{$servidor->id}", 'local');
        $documento->observacoes = $request->observacoes;
        $servidor->documentos()->save($documento);

        return redirect()
                ->route('admin.servidores.documentos.index', ['servidor' => $servidor])
                ->with('status', 'Documento incluído com sucesso!');
    }

    public function download(Servidor $servidor, DocumentoAssociado $documento)
    {
        return Storage::download($documento->caminho);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\DocumentoAssociado  $documento
     * @return \Illuminate\Http\Response
     */
    public function edit(Servidor $servidor, DocumentoAssociado $documento)
    {
        $tipos = TipoDocumento::orderBy('nome')->get();

        return view('admin.servidores.documentos-associados.edit', [
            'tipos' => $tipos,
            'documento' => $documento,
            'servidor' => $servidor,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\DocumentoAssociado  $documento
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Servidor $servidor, DocumentoAssociado $documento)
    {
        $this->validate($request, [
            'arquivo' => [
                'nullable',
                'file',
                implode(',', config('app.mimetypes-documentos-associados'))
            ],
            'tipo' => 'required|numeric',
            'observacoes' => 'nullable|max:100',
        ]);

        $tipo = TipoDocumento::find($request->tipo);
        $documento->tipo()->associate($tipo);
        $documento->observacoes = $request->observacoes;

        if ($request->arquivo) {
            Storage::delete($documento->caminho);

            $documento->caminho = $request->file('arquivo')->store("documentos-associados/{$servidor->id}", 'local');
        }

        $documento->save();

        return redirect()
            ->route('admin.servidores.documentos.index', ['servidor' => $servidor])
            ->with('status', 'Documento excluído com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\DocumentoAssociado  $documento
     * @return \Illuminate\Http\Response
     */
    public function destroy(Servidor $servidor, DocumentoAssociado $documento)
    {
        Storage::delete($documento->caminho);

        $documento->delete();

        return redirect()
            ->route('admin.servidores.documentos.index', ['servidor' => $servidor])
            ->with('status', 'Documento excluído com sucesso!');
    }
}
