<?php

namespace App\Http\Controllers\Admin;

use App\Enums\ResultadoProcessoJuridicoEnum;
use App\Http\Controllers\Controller;
use App\Http\Requests\ProcessoJuridicoStoreRequest;
use App\Http\Requests\ProcessoJuridicoUpdateRequest;
use App\Models\AgendamentoAtendimento;
use App\Models\CompetenciaProcesso;
use App\Models\Dependente;
use App\Models\Funcionario;
use App\Models\ProcessoJuridico;
use App\Models\Servidor;
use App\Models\TipoAcaoProcesso;
use Illuminate\Http\Request;

class ProcessoJuridicoController extends Controller
{
    public function __construct()
    {
        $this->middleware(['role:admin']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index($pessoa, $tipo)
    {
        $pessoa = $this->buscaPessoaPorTipoId($tipo, $pessoa);

        $processos = $pessoa->processosJuridicos()
            ->orderBy('id', 'desc')
            ->paginate();

        return view('admin.processos-juridicos.index', [
            'processos' => $processos,
            'pessoa' => $pessoa
        ]);
    }


    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request, $pessoa, $tipo)
    {
        $pessoa = $this->buscaPessoaPorTipoId($tipo, $pessoa);

        return view('admin.processos-juridicos.create', [
            'competencias' => CompetenciaProcesso::all(),
            'tiposAcoes' => TipoAcaoProcesso::all(),
            'resultados' => ResultadoProcessoJuridicoEnum::getAll(),
            'pessoa' => $pessoa,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(ProcessoJuridicoStoreRequest $request, $pessoa, $tipo)
    {
        $pessoa = $this->buscaPessoaPorTipoId($tipo, $pessoa);
        $processo = new ProcessoJuridico($request->all());
        $pessoa->processosJuridicos()->save($processo);

        return redirect()->route('admin.processos-juridicos.index', [
            'pessoa' => $pessoa,
            'tipo' => $tipo,
            'agendamento' => $request->agendamento
        ])->with('status', 'Processo incluído com sucesso!');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\ProcessoJuridico  $processo
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, $pessoa, $tipo, $id)
    {
        $processo = ProcessoJuridico::findOrFail($id);
        $atendimentos = $processo->atendimentos()->orderBy('id', 'desc')->get();

        if ($request->has('agendamento')) {
            $agendamento = AgendamentoAtendimento::find($request->agendamento);
        }

        return view('admin.processos-juridicos.edit', [
            'processo' => $processo,
            'competencias' => CompetenciaProcesso::all(),
            'tiposAcoes' => TipoAcaoProcesso::all(),
            'resultados' => ResultadoProcessoJuridicoEnum::getAll(),
            'agendamento' => $agendamento ?? null,
            'atendimentos' => $atendimentos,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\AtendimentoJuridico  $processo
     * @return \Illuminate\Http\Response
     */
    public function update(ProcessoJuridicoUpdateRequest $request, $pessoa, $tipo, $id)
    {
        $processo = ProcessoJuridico::findOrFail($id);
        $processo->fill($request->all());
        $processo->save();

        return redirect()->route('admin.processos-juridicos.index', [
            'pessoa' => $pessoa,
            'tipo' => $tipo,
            'agendamento' => $request->agendamento
        ])->with('status', 'Processo editado com sucesso!');
    }

    private function buscaPessoaPorTipoId($tipo, $id)
    {
        if ($tipo == 'funcionario') {
            $pessoa = Funcionario::findOrFail($id);
        }

        if ($tipo == 'associado') {
            $pessoa = Servidor::findOrFail($id);
        }

        if ($tipo == 'dependente') {
            $pessoa = Dependente::findOrFail($id);
        }


        return $pessoa;
    }

    public function destroy($pessoa, $tipo, $id)
    {
        $processo = ProcessoJuridico::findOrFail($id);
        $processo->delete();

        return redirect()->route('admin.processos-juridicos.index', ['pessoa' => $pessoa, 'tipo' => $tipo])
            ->with('status', 'Processo excluído com sucesso!');
    }
}
