<?php

namespace App\Http\Controllers\Admin;

use App\Enums\StatusMensalidadeNivelEnum;
use App\Http\Controllers\Controller;
use App\Http\Requests\MensalidadeNivelDeleteRequest;
use App\Http\Requests\MensalidadeNivelStoreRequest;
use App\Http\Requests\MensalidadeNivelUpdateRequest;
use App\Models\MensalidadeNivel;
use App\Models\NivelSalarial;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MensalidadeNivelController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }
    
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $filtroStatus = $request->status
                            ? StatusMensalidadeNivelEnum::make($request->status)
                            : StatusMensalidadeNivelEnum::todos();
        
        $mensalidades = MensalidadeNivel::with('nivelSalarial')
                                        ->comStatus($filtroStatus)
                                        ->orderBy('nivel_salarial_id')
                                        ->orderBy('inicio_vigencia', 'desc')
                                        ->paginate(20);
        $statuses = StatusMensalidadeNivelEnum::getAll();

        return view('admin.mensalidades.index', [
            'mensalidades' => $mensalidades,
            'statuses' => $statuses,
            'filtroStatus' => $filtroStatus,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.mensalidades.create', ['niveis' => NivelSalarial::all()]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(MensalidadeNivelStoreRequest $request)
    {
        $mensalidadeVigente = MensalidadeNivel::vigente($request->nivelSalarial);
        $mensalidade = new MensalidadeNivel($request->except('mensalidadeVigente', 'nivelSalarial'));

        DB::beginTransaction();

        $mensalidade->save();

        if ($mensalidadeVigente) {
            $mensalidadeVigente->termino_vigencia = $mensalidade->inicio_vigencia->subDay();
            $mensalidadeVigente->save();
        }

        DB::commit();

        return redirect()
                ->route('admin.mensalidades.index')
                ->with('status', 'Mensalidade incluída com sucesso!');
    }

    public function edit(MensalidadeNivel $mensalidade)
    {
        $niveis = NivelSalarial::all();
        return view('admin.mensalidades.edit', ['mensalidade' => $mensalidade, 'niveis' => $niveis]);
    }

    public function update(MensalidadeNivelUpdateRequest $request, MensalidadeNivel $mensalidade)
    {
        $mensalidade->fill($request->except('mensalidadeAnterior', 'nivelSalarial'));
        $mensalidade->save();

        return redirect()
            ->route('admin.mensalidades.index')
            ->with('status', 'Mensalidade editada com sucesso!');
    }

    public function destroy(MensalidadeNivelDeleteRequest $request, MensalidadeNivel $mensalidade)
    {
        DB::beginTransaction();
        
        $mensalidade->delete();

        if ($mensalidade->termino_vigencia == null) {
            $request->mensalidadeAnterior->termino_vigencia = null;
            $request->mensalidadeAnterior->save();
        }

        DB::commit();

        return redirect()
            ->route('admin.mensalidades.index')
            ->with('status', 'Mensalidade excluída com sucesso!');
    }
}
