<?php

namespace App\Http\Controllers\Admin;

use App\Enums\DiaSemanaEnum;
use App\Exceptions\ValidationException;
use App\Http\Controllers\Controller;
use App\Models\HorarioTrabalho;
use App\Models\ServicoAtendimento;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class HorarioTrabalhoServicoAtendimentoController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(ServicoAtendimento $servico)
    {
        $horarios = $servico->horariosTrabalho()->get()->groupBy('dia_semana')->toArray();

        return view('admin.horarios-trabalho.index', [
            'horarios' => $horarios,
            'diasSemana' => DiaSemanaEnum::getAll(),
            'trabalhavel' => 'Serviço de Atendimento',
            'routes' => $this->routes($servico),
        ]);
    }

    private function routes($servico, $horario = null)
    {
        return [
            'trabalhavel' => [
                'key' => 'servico',
                'index' => ['route' => 'admin.servicos-atendimentos.index'],
                'edit' =>  ['route' => 'admin.servicos-atendimentos.edit', 'params' => ['servico' => $servico]],
            ],
            'horario' => [
                'index'  => ['route' => 'admin.servicos-atendimentos.horarios-trabalho.index', 'params' => ['servico' => $servico]],
                'create' => ['route' => 'admin.servicos-atendimentos.horarios-trabalho.create', 'params' => ['servico' => $servico]],
                'edit'   => ['route' => 'admin.servicos-atendimentos.horarios-trabalho.edit', 'params' => ['servico' => $servico]],
                'store'  => ['route' => 'admin.servicos-atendimentos.horarios-trabalho.store', 'params' => ['servico' => $servico]],
                'update'  => ['route' => 'admin.servicos-atendimentos.horarios-trabalho.update', 'params' => ['servico' => $servico, 'horario' => $horario]],
                'destroy'  => ['route' => 'admin.servicos-atendimentos.horarios-trabalho.destroy', 'params' => ['servico' => $servico]],
            ],
        ];
    }

    private function valida($horario)
    {
        if (!$horario[0]['inicio'] && $horario[1]['inicio']) {
            throw new ValidationException('Não é possível preencher o segundo período sem preencher o primeiro período');
        }

        if ($horario[0]['inicio'] && $horario[0]['fim'] && $horario[1]['inicio'] && $horario[1]['fim']) {
            $inicio1 = Carbon::createFromFormat('H:i', $horario[0]['inicio']);
            $fim1 = Carbon::createFromFormat('H:i', $horario[0]['fim']);
            $inicio2 = Carbon::createFromFormat('H:i', $horario[1]['inicio']);

            if ($inicio2->lessThanOrEqualTo($inicio1) || $inicio2->lessThanOrEqualTo($fim1)) {
                throw new ValidationException('Início do segundo período deve ser posterior ao término do primeiro');
            }
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, ServicoAtendimento $servico)
    {
        $horarios = $request->horarios;

        DB::beginTransaction();

        $servico->horariosTrabalho()->delete();
        
        foreach ($horarios as $diaSemana => $h) {
            if (!$h[0]['inicio']) {
                continue;
            }

            $this->valida($h);

            $turno1 = new HorarioTrabalho();
            $turno1->inicio = $h[0]['inicio'];
            $turno1->fim = $h[0]['fim'];

            $turno1->dia_semana = DiaSemanaEnum::make($diaSemana)->getName();
            $turno1->trabalhavel()->associate($servico);
            $turno1->eValido();
            $turno1->save();

            if (! $h[1]['inicio']) {
                continue;
            }

            $turno2 = new HorarioTrabalho();
            $turno2->inicio = $h[1]['inicio'];
            $turno2->fim = $h[1]['fim'];
            $turno2->dia_semana = DiaSemanaEnum::make($diaSemana)->getName();
            $turno2->trabalhavel()->associate($servico);
            $turno2->eValido();
            $turno2->save();
        }

        DB::commit();

        return redirect()
                ->back()
                ->with('status', 'Horário de trabalho atualizado com sucesso!');
    }
}
