<?php

namespace App\Http\Controllers\Admin;

use App\Enums\DiaSemanaEnum;
use App\Exceptions\ValidationException;
use App\Http\Controllers\Controller;
use App\Models\HorarioTrabalho;
use App\Models\PrestadorServico;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class HorarioTrabalhoPrestadorController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(PrestadorServico $prestador)
    {
        $horarios = $prestador->horariosTrabalho()->get()->groupBy('dia_semana')->toArray();

        return view('admin.horarios-trabalho.index', [
            'horarios' => $horarios,
            'diasSemana' => DiaSemanaEnum::getAll(),
            'trabalhavel' => 'Prestador de serviço',
            'routes' => $this->routes($prestador),
        ]);
    }

    private function routes($prestador, $horario = null)
    {
        return [
            'trabalhavel' => [
                'key' => 'prestador',
                'index' => ['route' => 'admin.prestadores-servicos.index'],
                'edit' =>  ['route' => 'admin.prestadores-servicos.edit', 'params' => ['prestador' => $prestador]],
            ],
            'horario' => [
                'index'  => ['route' => 'admin.prestadores-servicos.horarios-trabalho.index', 'params' => ['prestador' => $prestador]],
                'create' => ['route' => 'admin.prestadores-servicos.horarios-trabalho.create', 'params' => ['prestador' => $prestador]],
                'edit'   => ['route' => 'admin.prestadores-servicos.horarios-trabalho.edit', 'params' => ['prestador' => $prestador]],
                'store'  => ['route' => 'admin.prestadores-servicos.horarios-trabalho.store', 'params' => ['prestador' => $prestador]],
                'update'  => ['route' => 'admin.prestadores-servicos.horarios-trabalho.update', 'params' => ['prestador' => $prestador, 'horario' => $horario]],
                'destroy'  => ['route' => 'admin.prestadores-servicos.horarios-trabalho.destroy', 'params' => ['prestador' => $prestador]],
            ],
        ];
    }

    private function valida($horario)
    {
        if (!$horario[0]['inicio'] && $horario[1]['inicio']) {
            throw new ValidationException('Não é possível preencher o segundo período sem preencher o primeiro período');
        }

        if ($horario[0]['inicio'] && $horario[0]['fim'] && $horario[1]['inicio'] && $horario[1]['fim']) {
            $inicio1 = Carbon::createFromFormat('H:i', $horario[0]['inicio']);
            $fim1 = Carbon::createFromFormat('H:i', $horario[0]['fim']);
            $inicio2 = Carbon::createFromFormat('H:i', $horario[1]['inicio']);

            if ($inicio2->lessThanOrEqualTo($inicio1) || $inicio2->lessThanOrEqualTo($fim1)) {
                throw new ValidationException('Início do segundo período deve ser posterior ao término do primeiro');
            }
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, PrestadorServico $prestador)
    {
        $horarios = $request->horarios;

        DB::beginTransaction();

        $prestador->horariosTrabalho()->delete();
        
        foreach ($horarios as $diaSemana => $h) {
            if (!$h[0]['inicio']) {
                continue;
            }

            $this->valida($h);

            $turno1 = new HorarioTrabalho();
            $turno1->inicio = $h[0]['inicio'];
            $turno1->fim = $h[0]['fim'];

            $turno1->dia_semana = DiaSemanaEnum::make($diaSemana)->getName();
            $turno1->trabalhavel()->associate($prestador);
            $turno1->eValido();
            $turno1->save();

            if (! $h[1]['inicio']) {
                continue;
            }

            $turno2 = new HorarioTrabalho();
            $turno2->inicio = $h[1]['inicio'];
            $turno2->fim = $h[1]['fim'];
            $turno2->dia_semana = DiaSemanaEnum::make($diaSemana)->getName();
            $turno2->trabalhavel()->associate($prestador);
            $turno2->eValido();
            $turno2->save();
        }

        DB::commit();

        return redirect()
                ->back()
                ->with('status', 'Horário de trabalho atualizado com sucesso!');
    }
}
