<?php

namespace App\Http\Controllers\Admin\Funcionario;

use App\Enums\SexoEnum;
use App\Http\Controllers\Controller;
use App\Http\Requests\FuncionarioStoreRequest;
use App\Http\Requests\FuncionarioUpdateRequest;
use App\Models\Funcionario;
use App\Models\Logradouro\Estado;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class FuncionarioController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $termo = $request->busca;
        $funcionarios = Funcionario::with('contatos')
            ->buscaPorTermo($termo)
            ->paginate();

        return view('admin.funcionarios.index', [
            'funcionarios' => $funcionarios,
            'termo' => $termo
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $sexos = SexoEnum::getAll();
        $estados = Estado::orderBy('ufe_sg')->select('ufe_sg')->get();
        return view('admin.funcionarios.create', [
            'sexos' => $sexos,
            'estados' => $estados,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(FuncionarioStoreRequest $request)
    {
        $funcionario = new Funcionario($request->only('nome', 'funcao', 'rg', 'cnh', 'cpf', 'data_nascimento', 'observacoes'));
        $funcionario->sexo = SexoEnum::make((int) $request->sexo);

        DB::beginTransaction();

        $funcionario->save();
        $funcionario->saveTelefones($request->telefone);
        $funcionario->saveEmails($request->email ?? []);

        if ($request->cep) {
            $funcionario->endereco->fill($request->only('cep', 'logradouro', 'numero', 'cidade', 'bairro', 'estado'));
            $funcionario->endereco->observacoes = $request->observacoes_endereco;
            $funcionario->endereco->save();
        }

        DB::commit();

        return redirect()
            ->route('admin.funcionarios.edit', ['funcionario' => $funcionario])
            ->with('sucesso', true)
            ->with('status', 'Funcionário incluído com sucesso!');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Funcionario  $funcionario
     * @return \Illuminate\Http\Response
     */
    public function edit(Funcionario $funcionario)
    {
        $sexos = SexoEnum::getAll();
        $estados = Estado::orderBy('ufe_sg')->select('ufe_sg')->get();
        return view('admin.funcionarios.edit', [
            'funcionario' => $funcionario,
            'sexos' => $sexos,
            'estados' => $estados,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Funcionario  $funcionario
     * @return \Illuminate\Http\Response
     */
    public function update(FuncionarioUpdateRequest $request, Funcionario $funcionario)
    {
        $funcionario->fill($request->only('nome', 'funcao', 'rg', 'cpf', 'cnh', 'data_nascimento', 'observacoes'));
        $funcionario->sexo = SexoEnum::make((int) $request->sexo);

        $funcionario->save();
        $funcionario->saveTelefones($request->telefone ?? []);
        $funcionario->saveEmails($request->email ?? []);

        if ($request->cep) {
            $funcionario->endereco->fill($request->only('cep', 'logradouro', 'numero', 'cidade', 'bairro', 'estado'));
            $funcionario->endereco->observacoes = $request->observacoes_endereco;
            $funcionario->endereco->save();
        } else {
            $funcionario->endereco->delete();
        }

        DB::commit();

        return redirect()
            ->back()
            ->with('sucesso', true)
            ->with('status', 'Funcionário editado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Funcionario  $funcionario
     * @return \Illuminate\Http\Response
     */
    public function destroy(Funcionario $funcionario)
    {
        $funcionario->delete();

        return redirect()->back()
            ->with('status', 'Funcionário excluído com sucesso!');
    }
}
