<?php

namespace App\Http\Controllers\Admin;

use App\Enums\TipoBeneficiarioConvenioEnum;
use App\Enums\TipoConvenioEnum;
use App\Http\Controllers\Controller;
use App\Http\Requests\ConvenioStoreRequest;
use App\Http\Requests\ConvenioUpdateRequest;
use App\Models\Convenio;
use App\Models\EspecialidadeConvenio;
use App\Models\Logradouro\Estado;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ConvenioController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $termo = $request->busca;
        $convenios = Convenio::buscaPorTermo($termo)
            ->paginate();

        return view('admin.convenios.index', [
            'convenios' => $convenios,
            'termo' => $termo
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $especialidades = EspecialidadeConvenio::all();
        $tipos = TipoConvenioEnum::getAll();
        $tiposBeneficiarios = TipoBeneficiarioConvenioEnum::getAll();
        $estados = Estado::orderBy('ufe_sg')->select('ufe_sg')->get();
        return view('admin.convenios.create', [
            'especialidades' => $especialidades,
            'tipos' => $tipos,
            'tiposBeneficiarios' => $tiposBeneficiarios,
            'estados' => $estados,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(ConvenioStoreRequest $request)
    {
        $convenio = new Convenio($request->all());
        $convenio->tipo = TipoConvenioEnum::make((int) $request->tipo);
        $convenio->tipo_beneficiario = TipoBeneficiarioConvenioEnum::make((int) $request->tipo_beneficiario);
        
        DB::beginTransaction();
        
        $convenio->save();
        $convenio->saveTelefones($request->telefone);

        if ($request->cep) {
            $convenio->endereco->fill($request->only('cep', 'logradouro', 'numero', 'cidade', 'bairro', 'estado'));
            $convenio->endereco->observacoes = $request->observacoes_endereco;
            $convenio->endereco->save();
        }

        DB::commit();

        return redirect()
                ->route('admin.convenios.index')
                ->with('Convênio incluído com sucesso!');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Convenio  $convenio
     * @return \Illuminate\Http\Response
     */
    public function edit(Convenio $convenio)
    {
        $especialidades = EspecialidadeConvenio::all();
        $tipos = TipoConvenioEnum::getAll();
        $tiposBeneficiarios = TipoBeneficiarioConvenioEnum::getAll();
        $estados = Estado::orderBy('ufe_sg')->select('ufe_sg')->get();
        return view('admin.convenios.edit', [
            'especialidades' => $especialidades,
            'tipos' => $tipos,
            'tiposBeneficiarios' => $tiposBeneficiarios,
            'estados' => $estados,
            'convenio' => $convenio
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Convenio  $convenio
     * @return \Illuminate\Http\Response
     */
    public function update(ConvenioUpdateRequest $request, Convenio $convenio)
    {
        $convenio->fill($request->all());
        $convenio->tipo = TipoConvenioEnum::make((int) $request->tipo);
        $convenio->tipo_beneficiario = TipoBeneficiarioConvenioEnum::make((int) $request->tipo_beneficiario);
        $convenio->save();
        $convenio->saveTelefones($request->telefone ?? []);

        if ($request->cep) {
            $convenio->endereco->fill($request->only('cep', 'logradouro', 'numero', 'cidade', 'bairro', 'estado'));
            $convenio->endereco->observacoes = $request->observacoes_endereco;
            $convenio->endereco->save();
        } else {
            $convenio->endereco->delete();
        }

        DB::commit();

        return redirect()
            ->route('admin.convenios.index')
            ->with('Convênio editado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Convenio  $convenio
     * @return \Illuminate\Http\Response
     */
    public function destroy(Convenio $convenio)
    {
        $convenio->delete();

        return redirect()
            ->route('admin.convenios.index')
            ->with('Convênio excluído com sucesso!');
    }
}
