<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Cargo;
use App\Models\NivelSalarial;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class CargoController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin']);
    }
    
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $cargos = Cargo::search(request('busca'))
                    ->paginate();

        return view('admin.cargos.index', ['cargos' => $cargos, 'busca' => request('busca')]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $niveis = NivelSalarial::orderBy('nome')->get();
        return view('admin.cargos.create', ['niveis' => $niveis]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'nome' => 'required|max:255|unique:cargos,nome',
            'nivel_salarial_id' => 'required|numeric',
        ]);

        $cargo = new Cargo($request->all());
        $cargo->nivelSalarial()->associate(NivelSalarial::find($request->nivel_salarial_id));
        $cargo->save();

        return redirect()
                ->route('admin.cargos.index')
                ->with('status', 'Cargo incluído com sucesso!');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Cargo  $cargo
     * @return \Illuminate\Http\Response
     */
    public function edit(Cargo $cargo)
    {
        $niveis = NivelSalarial::orderBy('nome')->get();
        return view('admin.cargos.edit', ['cargo' => $cargo, 'niveis' => $niveis]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Cargo  $cargo
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Cargo $cargo)
    {
        $this->validate($request, [
            'nome' => ['required', 'max:255', Rule::unique('cargos')->ignore($cargo)],
            'nivel_salarial_id' => 'required|numeric',
        ]);

        $cargo->fill($request->all());
        $cargo->nivelSalarial()->associate(NivelSalarial::find($request->nivel_salarial_id));
        $cargo->save();

        return redirect()
            ->route('admin.cargos.index')
            ->with('status', 'Cargo editado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Cargo  $cargo
     * @return \Illuminate\Http\Response
     */
    public function destroy(Cargo $cargo)
    {
        $cargo->delete();

        return redirect()
            ->route('admin.cargos.index')
            ->with('status', 'Cargo excluído com sucesso!');
    }
}
