<?php

namespace App\Http\Controllers\Admin;

use App\Enums\ResultadoProcessoJuridicoEnum;
use App\Models\AnexoProcessoJuridico;
use App\Http\Controllers\Controller;
use App\Models\CompetenciaProcesso;
use App\Models\Dependente;
use App\Models\Funcionario;
use App\Models\ProcessoJuridico;
use App\Models\Servidor;
use App\Models\TipoAcaoProcesso;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class AnexoProcessoJuridicoController extends Controller
{
    public function __construct()
    {
        $this->middleware(['role:admin']);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request, $idPessoa, $tipo, $idProcesso)
    {
        $processo = ProcessoJuridico::findOrFail($idProcesso);
        return view('admin.processos-juridicos.anexos.create', [
            'processo' => $processo
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, $idPessoa, $tipo, $idProcesso)
    {
        $this->validate($request, [
            'descricao' => 'nullable|max:150',
            'arquivo' => 'required|file'
        ]);

        $processo = ProcessoJuridico::findOrFail($idProcesso);
        $anexo = new AnexoProcessoJuridico($request->all());
        $anexo->caminho = $request->arquivo->store("processos/{$processo->id}/anexos", 'local');
        $processo->anexos()->save($anexo);

        return redirect()->route('admin.processos-juridicos.edit', ['id' => $idProcesso, 'tipo' => $tipo, 'pessoa' => $idPessoa])
                ->with('status', 'Anexo de processo incluído com sucesso!');
    }

    public function download($idPessoa, $tipo, $idProcesso, $id)
    {
        $processo = ProcessoJuridico::findOrFail($idProcesso);
        $anexo = AnexoProcessoJuridico::findOrFail($id);

        return Storage::disk('local')->download($anexo->caminho, "anexo-processo-{$processo->numero}");
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\ProcessoJuridico  $processo
     * @return \Illuminate\Http\Response
     */
    public function edit($idPessoa, $tipo, $idProcesso, $id)
    {
        $processo = ProcessoJuridico::findOrFail($idProcesso);
        $anexo = AnexoProcessoJuridico::findOrFail($id);

        return view('admin.processos-juridicos.anexos.edit', [
            'processo' => $processo,
            'anexo' => $anexo,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\AtendimentoJuridico  $processo
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $idPessoa, $tipo, $idProcesso, $id)
    {
        $this->validate($request, [
            'descricao' => 'nullable|max:150',
            'arquivo' => 'nullable|file'
        ]);

        $processo = ProcessoJuridico::findOrFail($idProcesso);
        $anexo = AnexoProcessoJuridico::findOrFail($id);
        $anexo->fill($request->all());

        if ($request->arquivo) {
            Storage::disk('local')->delete($anexo->caminho);
            $anexo->caminho = $request->arquivo->store("processos/{$processo->id}/anexos", 'local');
            
            info("Arquivo de anexo de processo atualizado", [
                'processo' => $idProcesso,
                'anexo' => $id,
                'arquivoNovo' => $anexo->caminho,
            ]);
        }

        $anexo->save();

        return redirect()->route('admin.processos-juridicos.edit', ['id' => $idProcesso, 'tipo' => $tipo, 'pessoa' => $idPessoa])
                ->with('status', 'Anexo de processo editado com sucesso!');
    }

    public function destroy($idPessoa, $tipo, $idProcesso, $id)
    {
        $anexo = AnexoProcessoJuridico::findOrFail($id);
        $anexo->delete();

        Storage::disk('local')->delete($anexo->caminho);

        info("Anexo de processo excluído junto de seu arquivo", [
            'processo' => $idProcesso,
            'anexo' => $id,
            'arquivo' => $anexo->caminho,
        ]);

        return redirect()->route('admin.processos-juridicos.edit', ['id' => $idProcesso, 'tipo' => $tipo, 'pessoa' => $idPessoa])
            ->with('status', 'Anexo de processo excluído com sucesso!');
    }
}
