<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\GoogleCalendarEventStoreRequest;
use App\Http\Requests\GoogleCalendarEventUpdateRequest;
use App\Models\Contato;
use App\Models\GoogleCalendarEvent as Event;
use Carbon\Carbon;
use GuzzleHttp\Exception\ConnectException;
use Illuminate\Http\Request;

class AgendaDiretoriaController extends Controller
{
    public function __construct()
    {
        $this->middleware(['role:admin']);
    }
    
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        return view('admin.agenda-diretoria.index');
    }

    public function load(Request $request)
    {
        $start = $request->has('start') ? Carbon::make($request->start) : null;
        $end = $request->has('end') ? Carbon::make($request->end) : null;

        try {
            $events =  Event::get($start, $end);
        } catch (ConnectException $e) {
            return response()->json([
                'erro' => 'Ocorreu um erro de comunicação com o Google Calendar e não foi possível obter os eventos. Verifique sua conexão com a internet.'
            ], 404);
        }

        $formatted = [];
        $events->pluck('googleEvent')->each(function ($event) use (&$formatted) {
            $formatted[] = [
                'title' => $event->summary,
                'start' => $event->start->dateTime,
                'end' => $event->end->dateTime,
                'url' => route('admin.agenda-diretoria.edit', ['evento' => $event->id]),
            ];
            return ['title' => $event->summary];
        });
        return $formatted;
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $emails = Contato::emails()->get()->pluck('valor')->unique();
        return view('admin.agenda-diretoria.create', ['emails' => $emails]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(GoogleCalendarEventStoreRequest $request)
    {
        $datas = explode(' - ', $request->data_inicio_fim);
        
        $evento = new Event();
        $evento->summary = $request->titulo;
        $evento->description = $request->descricao;
        $evento->location = $request->local;
        $evento->startDateTime = Carbon::createFromFormat('d/m/Y H:i', $datas[0]);
        $evento->endDateTime = Carbon::createFromFormat('d/m/Y H:i', $datas[1]);
        collect($request->convidados)->each(function ($convidado) use (&$evento) {
            $evento->addAttendee(['email' => $convidado]);
        });

        try {
            $evento->save();
        } catch (ConnectException $e) {
            return redirect()
                ->route('admin.agenda-diretoria.index');
        }

        return redirect()
                ->route('admin.agenda-diretoria.index')
                ->with('status', 'Evento incluído com sucesso!');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {
            $evento = Event::find($id);
        } catch (ConnectException $e) {
            return redirect()
                    ->route('admin.agenda-diretoria.index');
        }
        
        return view('admin.agenda-diretoria.edit', ['evento' => $evento->formatToView()]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(GoogleCalendarEventUpdateRequest $request, $id)
    {
        $datas = explode(' - ', $request->data_inicio_fim);

        try {
            $evento = Event::find($id);
            $evento->summary = $request->titulo;
            $evento->description = $request->descricao;
            $evento->location = $request->local;
            $evento->startDateTime = Carbon::createFromFormat('d/m/Y H:i', $datas[0]);
            $evento->endDateTime = Carbon::createFromFormat('d/m/Y H:i', $datas[1]);
        
            $evento->save();
        } catch (ConnectException $e) {
            return redirect()
                    ->route('admin.agenda-diretoria.index');
        }

        return redirect()
            ->route('admin.agenda-diretoria.index')
            ->with('status', 'Evento editado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $evento = Event::find($id);
            $evento->delete();
        } catch (ConnectException $e) {
            return redirect()
                ->route('admin.agenda-diretoria.index');
        }
        
        return redirect()
            ->route('admin.agenda-diretoria.index')
            ->with('status', 'Evento excluído com sucesso!');
    }
}
